unit LibJpegLosslless;

interface

uses
  Classes;

function drop(const ASrc, ADest, ADrop: TStream; const ALeft, ATop: Integer): Boolean;
function crop(const ASrc, ADest: TStream; const ALeft, ATop, ARight, ABottom: Integer): Boolean;

implementation

uses
  SysUtils,
  LibJpeg8,
  LibJpegTran,
  LibJpegErrorHandler,
  LibJpegInOutDataManager;

const
  JFALSE = JBOOL(0);
  JTRUE  = JBOOL(1);

function JpegLosslessTransform(
  const ASrc:  TStream;
  const ADest: TStream;
  const ADrop: TStream;
  const AOperation: JXFORM_CODE;
  const ACrop: string
): Boolean;
var
  crop_parse_result: boolean;
  srcinfo: jpeg_decompress_struct;
  dropinfo: jpeg_decompress_struct;
	dstinfo: jpeg_compress_struct;
  jsrcerr, jdsterr, jdroperr: jpeg_error_mgr;
  src_coef_arrays: jvirt_barray_ptr_ptr;
  dst_coef_arrays: jvirt_barray_ptr_ptr;
  copyoption: JCOPY_OPTION;
  transfoptions: jpeg_transform_info;
begin
  // Initialize structures
  FillChar(srcinfo, SizeOf(jpeg_decompress_struct), 0);
  FillChar(jsrcerr, SizeOf(jpeg_error_mgr), 0);

  FillChar(dropinfo, SizeOf(jpeg_decompress_struct), 0);
  FillChar(jdroperr, SizeOf(jpeg_error_mgr), 0);

  FillChar(dstinfo, SizeOf(jpeg_compress_struct), 0);
  FillChar(jdsterr, SizeOf(jpeg_error_mgr), 0);

  FillChar(transfoptions, SizeOf(jpeg_transform_info), 0);

  // Copy all extra markers from source file
	copyoption := JCOPYOPT_ALL;

  // Set up default JPEG parameters
	transfoptions.force_grayscale := JFALSE;

	transfoptions.crop := JFALSE;

  transfoptions.transform := AOperation;

  transfoptions.perfect := JFALSE;

  //transfoptions.trim := JTRUE;

  try
    // Initialize the JPEG decompression object with default error handling
    srcinfo.err := jpeg_std_error(@jsrcerr);
    jsrcerr.error_exit := libjpeg_error_exit;
    jsrcerr.output_message := libjpeg_output_message;
    jpeg_create_decompress(@srcinfo);

    if transfoptions.transform = JXFORM_DROP then begin
      dropinfo.err := jpeg_std_error(@jdroperr);
      jdroperr.error_exit := libjpeg_error_exit;
      jdroperr.output_message := libjpeg_output_message;
      jpeg_create_decompress(@dropinfo);
    end;

    // Initialize the JPEG compression object with default error handling
    dstinfo.err := jpeg_std_error(@jdsterr);
    dstinfo.err.error_exit := libjpeg_error_exit;
    dstinfo.err.output_message := libjpeg_output_message;
    jpeg_create_compress(@dstinfo);

    // crop&drop option
    crop_parse_result := jtransform_parse_crop_spec(@transfoptions, PAnsiChar(ACrop));

    if not crop_parse_result then begin
      raise Exception.CreateFmt('Bogus crop argument %s', [ACrop]);
    end;

    if ( (not crop_parse_result) or
           ( (transfoptions.transform = JXFORM_DROP) and
             ( (transfoptions.crop_width_set <> JCROP_UNSET) or
               (transfoptions.crop_height_set <> JCROP_UNSET)
             )
           )
       )
    then begin
      raise Exception.CreateFmt('Bogus drop argument %s', [ACrop]);
    end;

    srcinfo.src := srcinfo.mem^.alloc_small(@srcinfo, JPOOL_PERMANENT, SizeOf(TJpeg8InPutDataManager) );
    with PJpeg8InPutDataManager(srcinfo.src)^ do begin
      jpeg_src_mgr.init_source := libjpeg_init_source;
      jpeg_src_mgr.fill_input_buffer := libjpeg_fill_input_buffer;
      jpeg_src_mgr.skip_input_data := libjpeg_skip_input_data;
      jpeg_src_mgr.resync_to_restart := nil; // use default method
      jpeg_src_mgr.term_source := libjpeg_term_source;
    end;
    srcinfo.client_data := @ASrc;

    // Enable saving of extra markers that we want to copy
    jcopy_markers_setup(@srcinfo, copyoption);

    // Read the file header
    jpeg_read_header(@srcinfo, TRUE);

    if transfoptions.transform = JXFORM_DROP then begin
      dropinfo.src := dropinfo.mem^.alloc_small(@dropinfo, JPOOL_PERMANENT, SizeOf(TJpeg8InPutDataManager) );
      with PJpeg8InPutDataManager(dropinfo.src)^ do begin
        jpeg_src_mgr.init_source := libjpeg_init_source;
        jpeg_src_mgr.fill_input_buffer := libjpeg_fill_input_buffer;
        jpeg_src_mgr.skip_input_data := libjpeg_skip_input_data;
        jpeg_src_mgr.resync_to_restart := nil; // use default method
        jpeg_src_mgr.term_source := libjpeg_term_source;
      end;
      dropinfo.client_data := @ADrop;

      jpeg_read_header(@dropinfo, TRUE);

      transfoptions.crop_width := dropinfo.image_width;
      transfoptions.crop_width_set := JCROP_POS;
      transfoptions.crop_height := dropinfo.image_height;
      transfoptions.crop_height_set := JCROP_POS;
      transfoptions.drop_ptr := @dropinfo;
    end;

    // Any space needed by a transform option must be requested before
    // jpeg_read_coefficients so that memory allocation will be done right

    // Prepare transformation workspace
    // Fails right away if perfect flag is TRUE and transformation is not perfect
    if not jtransform_request_workspace(@srcinfo, @transfoptions) then begin
      raise Exception.Create('Transformation is not perfect');
    end;

    // Read source file as DCT coefficients
    src_coef_arrays := jpeg_read_coefficients(@srcinfo);

    if transfoptions.transform = JXFORM_DROP then begin
      transfoptions.drop_coef_arrays := jpeg_read_coefficients(@dropinfo);
    end;

    // Initialize destination compression parameters from source values
    jpeg_copy_critical_parameters(@srcinfo, @dstinfo);

    // Adjust destination parameters if required by transform options;
    // also find out which set of coefficient arrays will hold the output
    dst_coef_arrays := jtransform_adjust_parameters(@srcinfo, @dstinfo, src_coef_arrays, @transfoptions);

    dstinfo.dest := dstinfo.mem^.alloc_small(@dstinfo, JPOOL_PERMANENT, SizeOf(TJpeg8OutPutDataManager) );
    with PJpeg8OutPutDataManager(dstinfo.dest)^ do begin
      jpeg_dest_mgr.init_destination := libjpeg_init_destination;
      jpeg_dest_mgr.empty_output_buffer := libjpeg_empty_output_buffer;
      jpeg_dest_mgr.term_destination := libjpeg_term_destination;
    end;
    dstinfo.client_data := @ADest;

    // Start compressor (note no image data is actually written here)
    jpeg_write_coefficients(@dstinfo, dst_coef_arrays);

    // Copy to the output file any extra markers that we want to preserve
    jcopy_markers_execute(@srcinfo, @dstinfo, copyoption);

    // Execute image transformation, if any
    jtransform_execute_transform(@srcinfo, @dstinfo, src_coef_arrays, @transfoptions);

    // Finish compression and release memory
    jpeg_finish_compress(@dstinfo);
    jpeg_destroy_compress(@dstinfo);

    if transfoptions.transform = JXFORM_DROP then begin
      jpeg_finish_decompress(@dropinfo);
      jpeg_destroy_decompress(@dropinfo);
    end;

    jpeg_finish_decompress(@srcinfo);
    jpeg_destroy_decompress(@srcinfo);
  except
    jpeg_destroy_compress(@dstinfo);

    if transfoptions.transform = JXFORM_DROP then begin
      jpeg_destroy_decompress(@dropinfo);
    end;

		jpeg_destroy_decompress(@srcinfo);
  end;
  Result := True;
end;

function crop(const ASrc, ADest: TStream; const ALeft, ATop, ARight,
  ABottom: Integer): Boolean;
var
  VCrop: string;
begin
  if not InitLibJpeg8() then begin
    raise Exception.Create('Can''t initialize ' + LIB_JPEG_NAME + '!');
  end;

  if not InitLibJpegTran() then begin
    raise Exception.Create('Can''t initialize ' + LIB_JPEG_TRAN_NAME + '!');
  end;

  VCrop := IntToStr(ARight - ALeft) + 'x' + IntToStr(ABottom - ATop) + '+' +
    IntToStr(ALeft) + '+' + IntToStr(ATop);

  Result := JpegLosslessTransform(ASrc, ADest, nil, JXFORM_NONE, VCrop);
end;

function drop(const ASrc, ADest, ADrop: TStream; const ALeft, ATop: Integer): Boolean;
var
  VCrop: string;
begin
  if not InitLibJpeg8() then begin
    raise Exception.Create('Can''t initialize ' + LIB_JPEG_NAME + '!');
  end;

  if not InitLibJpegTran() then begin
    raise Exception.Create('Can''t initialize ' + LIB_JPEG_TRAN_NAME + '!');
  end;

  VCrop := '+' + IntToStr(ALeft) + '+' + IntToStr(ATop);

  Result := JpegLosslessTransform(ASrc, ADest, ADrop, JXFORM_DROP, VCrop);
end;

end.
