unit LibPng15;

(* png.h - header file for PNG reference library
 *
 * libpng version 1.5.10 - March 29, 2012
 * Copyright (c) 1998-2012 Glenn Randers-Pehrson
 * (Version 0.96 Copyright (c) 1996, 1997 Andreas Dilger)
 * (Version 0.88 Copyright (c) 1995, 1996 Guy Eric Schalnat, Group 42, Inc.)
 *
 * This code is released under the libpng license (See LICENSE, below)
 *
 * Authors and maintainers:
 *   libpng versions 0.71, May 1995, through 0.88, January 1996: Guy Schalnat
 *   libpng versions 0.89c, June 1996, through 0.96, May 1997: Andreas Dilger
 *   libpng versions 0.97, January 1998, through 1.5.10 - March 29, 2012: Glenn
 *   See also "Contributing Authors", below.
 *
 * Note about libpng version numbers:
 *
 *   Due to various miscommunications, unforeseen code incompatibilities
 *   and occasional factors outside the authors' control, version numbering
 *   on the library has not always been consistent and straightforward.
 *   The following table summarizes matters since version 0.89c, which was
 *   the first widely used release:
 *
 *    source                 png.h  png.h  shared-lib
 *    version                string   int  version
 *    -------                ------ -----  ----------
 *    0.89c "1.0 beta 3"     0.89      89  1.0.89
 *    0.90  "1.0 beta 4"     0.90      90  0.90  [should have been 2.0.90]
 *    0.95  "1.0 beta 5"     0.95      95  0.95  [should have been 2.0.95]
 *    0.96  "1.0 beta 6"     0.96      96  0.96  [should have been 2.0.96]
 *    0.97b "1.00.97 beta 7" 1.00.97   97  1.0.1 [should have been 2.0.97]
 *    0.97c                  0.97      97  2.0.97
 *    0.98                   0.98      98  2.0.98
 *    0.99                   0.99      98  2.0.99
 *    0.99a-m                0.99      99  2.0.99
 *    1.00                   1.00     100  2.1.0 [100 should be 10000]
 *    1.0.0      (from here on, the   100  2.1.0 [100 should be 10000]
 *    1.0.1       png.h string is   10001  2.1.0
 *    1.0.1a-e    identical to the  10002  from here on, the shared library
 *    1.0.2       source version)   10002  is 2.V where V is the source code
 *    1.0.2a-b                      10003  version, except as noted.
 *    1.0.3                         10003
 *    1.0.3a-d                      10004
 *    1.0.4                         10004
 *    1.0.4a-f                      10005
 *    1.0.5 (+ 2 patches)           10005
 *    1.0.5a-d                      10006
 *    1.0.5e-r                      10100 (not source compatible)
 *    1.0.5s-v                      10006 (not binary compatible)
 *    1.0.6 (+ 3 patches)           10006 (still binary incompatible)
 *    1.0.6d-f                      10007 (still binary incompatible)
 *    1.0.6g                        10007
 *    1.0.6h                        10007  10.6h (testing xy.z so-numbering)
 *    1.0.6i                        10007  10.6i
 *    1.0.6j                        10007  2.1.0.6j (incompatible with 1.0.0)
 *    1.0.7beta11-14        DLLNUM  10007  2.1.0.7beta11-14 (binary compatible)
 *    1.0.7beta15-18           1    10007  2.1.0.7beta15-18 (binary compatible)
 *    1.0.7rc1-2               1    10007  2.1.0.7rc1-2 (binary compatible)
 *    1.0.7                    1    10007  (still compatible)
 *    1.0.8beta1-4             1    10008  2.1.0.8beta1-4
 *    1.0.8rc1                 1    10008  2.1.0.8rc1
 *    1.0.8                    1    10008  2.1.0.8
 *    1.0.9beta1-6             1    10009  2.1.0.9beta1-6
 *    1.0.9rc1                 1    10009  2.1.0.9rc1
 *    1.0.9beta7-10            1    10009  2.1.0.9beta7-10
 *    1.0.9rc2                 1    10009  2.1.0.9rc2
 *    1.0.9                    1    10009  2.1.0.9
 *    1.0.10beta1              1    10010  2.1.0.10beta1
 *    1.0.10rc1                1    10010  2.1.0.10rc1
 *    1.0.10                   1    10010  2.1.0.10
 *    1.0.11beta1-3            1    10011  2.1.0.11beta1-3
 *    1.0.11rc1                1    10011  2.1.0.11rc1
 *    1.0.11                   1    10011  2.1.0.11
 *    1.0.12beta1-2            2    10012  2.1.0.12beta1-2
 *    1.0.12rc1                2    10012  2.1.0.12rc1
 *    1.0.12                   2    10012  2.1.0.12
 *    1.1.0a-f                 -    10100  2.1.1.0a-f (branch abandoned)
 *    1.2.0beta1-2             2    10200  2.1.2.0beta1-2
 *    1.2.0beta3-5             3    10200  3.1.2.0beta3-5
 *    1.2.0rc1                 3    10200  3.1.2.0rc1
 *    1.2.0                    3    10200  3.1.2.0
 *    1.2.1beta1-4             3    10201  3.1.2.1beta1-4
 *    1.2.1rc1-2               3    10201  3.1.2.1rc1-2
 *    1.2.1                    3    10201  3.1.2.1
 *    1.2.2beta1-6            12    10202  12.so.0.1.2.2beta1-6
 *    1.0.13beta1             10    10013  10.so.0.1.0.13beta1
 *    1.0.13rc1               10    10013  10.so.0.1.0.13rc1
 *    1.2.2rc1                12    10202  12.so.0.1.2.2rc1
 *    1.0.13                  10    10013  10.so.0.1.0.13
 *    1.2.2                   12    10202  12.so.0.1.2.2
 *    1.2.3rc1-6              12    10203  12.so.0.1.2.3rc1-6
 *    1.2.3                   12    10203  12.so.0.1.2.3
 *    1.2.4beta1-3            13    10204  12.so.0.1.2.4beta1-3
 *    1.0.14rc1               13    10014  10.so.0.1.0.14rc1
 *    1.2.4rc1                13    10204  12.so.0.1.2.4rc1
 *    1.0.14                  10    10014  10.so.0.1.0.14
 *    1.2.4                   13    10204  12.so.0.1.2.4
 *    1.2.5beta1-2            13    10205  12.so.0.1.2.5beta1-2
 *    1.0.15rc1-3             10    10015  10.so.0.1.0.15rc1-3
 *    1.2.5rc1-3              13    10205  12.so.0.1.2.5rc1-3
 *    1.0.15                  10    10015  10.so.0.1.0.15
 *    1.2.5                   13    10205  12.so.0.1.2.5
 *    1.2.6beta1-4            13    10206  12.so.0.1.2.6beta1-4
 *    1.0.16                  10    10016  10.so.0.1.0.16
 *    1.2.6                   13    10206  12.so.0.1.2.6
 *    1.2.7beta1-2            13    10207  12.so.0.1.2.7beta1-2
 *    1.0.17rc1               10    10017  12.so.0.1.0.17rc1
 *    1.2.7rc1                13    10207  12.so.0.1.2.7rc1
 *    1.0.17                  10    10017  12.so.0.1.0.17
 *    1.2.7                   13    10207  12.so.0.1.2.7
 *    1.2.8beta1-5            13    10208  12.so.0.1.2.8beta1-5
 *    1.0.18rc1-5             10    10018  12.so.0.1.0.18rc1-5
 *    1.2.8rc1-5              13    10208  12.so.0.1.2.8rc1-5
 *    1.0.18                  10    10018  12.so.0.1.0.18
 *    1.2.8                   13    10208  12.so.0.1.2.8
 *    1.2.9beta1-3            13    10209  12.so.0.1.2.9beta1-3
 *    1.2.9beta4-11           13    10209  12.so.0.9[.0]
 *    1.2.9rc1                13    10209  12.so.0.9[.0]
 *    1.2.9                   13    10209  12.so.0.9[.0]
 *    1.2.10beta1-7           13    10210  12.so.0.10[.0]
 *    1.2.10rc1-2             13    10210  12.so.0.10[.0]
 *    1.2.10                  13    10210  12.so.0.10[.0]
 *    1.4.0beta1-5            14    10400  14.so.0.0[.0]
 *    1.2.11beta1-4           13    10211  12.so.0.11[.0]
 *    1.4.0beta7-8            14    10400  14.so.0.0[.0]
 *    1.2.11                  13    10211  12.so.0.11[.0]
 *    1.2.12                  13    10212  12.so.0.12[.0]
 *    1.4.0beta9-14           14    10400  14.so.0.0[.0]
 *    1.2.13                  13    10213  12.so.0.13[.0]
 *    1.4.0beta15-36          14    10400  14.so.0.0[.0]
 *    1.4.0beta37-87          14    10400  14.so.14.0[.0]
 *    1.4.0rc01               14    10400  14.so.14.0[.0]
 *    1.4.0beta88-109         14    10400  14.so.14.0[.0]
 *    1.4.0rc02-08            14    10400  14.so.14.0[.0]
 *    1.4.0                   14    10400  14.so.14.0[.0]
 *    1.4.1beta01-03          14    10401  14.so.14.1[.0]
 *    1.4.1rc01               14    10401  14.so.14.1[.0]
 *    1.4.1beta04-12          14    10401  14.so.14.1[.0]
 *    1.4.1                   14    10401  14.so.14.1[.0]
 *    1.4.2                   14    10402  14.so.14.2[.0]
 *    1.4.3                   14    10403  14.so.14.3[.0]
 *    1.4.4                   14    10404  14.so.14.4[.0]
 *    1.5.0beta01-58          15    10500  15.so.15.0[.0]
 *    1.5.0rc01-07            15    10500  15.so.15.0[.0]
 *    1.5.0                   15    10500  15.so.15.0[.0]
 *    1.5.1beta01-11          15    10501  15.so.15.1[.0]
 *    1.5.1rc01-02            15    10501  15.so.15.1[.0]
 *    1.5.1                   15    10501  15.so.15.1[.0]
 *    1.5.2beta01-03          15    10502  15.so.15.2[.0]
 *    1.5.2rc01-03            15    10502  15.so.15.2[.0]
 *    1.5.2                   15    10502  15.so.15.2[.0]
 *    1.5.3beta01-10          15    10503  15.so.15.3[.0]
 *    1.5.3rc01-02            15    10503  15.so.15.3[.0]
 *    1.5.3beta11             15    10503  15.so.15.3[.0]
 *    1.5.3 [omitted]
 *    1.5.4beta01-08          15    10504  15.so.15.4[.0]
 *    1.5.4rc01               15    10504  15.so.15.4[.0]
 *    1.5.4                   15    10504  15.so.15.4[.0]
 *    1.5.5beta01-08          15    10505  15.so.15.5[.0]
 *    1.5.5rc01               15    10505  15.so.15.5[.0]
 *    1.5.5                   15    10505  15.so.15.5[.0]
 *    1.5.6beta01-07          15    10506  15.so.15.6[.0]
 *    1.5.6rc01-03            15    10506  15.so.15.6[.0]
 *    1.5.6                   15    10506  15.so.15.6[.0]
 *    1.5.7beta01-05          15    10507  15.so.15.7[.0]
 *    1.5.7rc01-03            15    10507  15.so.15.7[.0]
 *    1.5.7                   15    10507  15.so.15.7[.0]
 *    1.5.8beta01             15    10508  15.so.15.8[.0]
 *    1.5.8rc01               15    10508  15.so.15.8[.0]
 *    1.5.8                   15    10508  15.so.15.8[.0]
 *    1.5.9beta01-02          15    10509  15.so.15.9[.0]
 *    1.5.9rc01               15    10509  15.so.15.9[.0]
 *    1.5.9                   15    10509  15.so.15.9[.0]
 *    1.5.10beta01-05         15    10510  15.so.15.10[.0]
 *    1.5.10                  15    10510  15.so.15.10[.0]
 *
 *   Henceforth the source version will match the shared-library major
 *   and minor numbers; the shared-library major version number will be
 *   used for changes in backward compatibility, as it is intended.  The
 *   PNG_LIBPNG_VER macro, which is not used within libpng but is available
 *   for applications, is an unsigned integer of the form xyyzz corresponding
 *   to the source version x.y.z (leading zeros in y and z).  Beta versions
 *   were given the previous public release number plus a letter, until
 *   version 1.0.6j; from then on they were given the upcoming public
 *   release number plus "betaNN" or "rcNN".
 *
 *   Binary incompatibility exists only when applications make direct access
 *   to the info_ptr or png_ptr members through png.h, and the compiled
 *   application is loaded with a different version of the library.
 *
 *   DLLNUM will change each time there are forward or backward changes
 *   in binary compatibility (e.g., when a new feature is added).
 *
 * See libpng-manual.txt or libpng.3 for more information.  The PNG
 * specification is available as a W3C Recommendation and as an ISO
 * Specification, <http://www.w3.org/TR/2003/REC-PNG-20031110/
 *)

(*
 * COPYRIGHT NOTICE, DISCLAIMER, and LICENSE:
 *
 * If you modify libpng you may insert additional notices immediately following
 * this sentence.
 *
 * This code is released under the libpng license.
 *
 * libpng versions 1.2.6, August 15, 2004, through 1.5.10, March 29, 2012, are
 * Copyright (c) 2004, 2006-2012 Glenn Randers-Pehrson, and are
 * distributed according to the same disclaimer and license as libpng-1.2.5
 * with the following individual added to the list of Contributing Authors:
 *
 *    Cosmin Truta
 *
 * libpng versions 1.0.7, July 1, 2000, through 1.2.5, October 3, 2002, are
 * Copyright (c) 2000-2002 Glenn Randers-Pehrson, and are
 * distributed according to the same disclaimer and license as libpng-1.0.6
 * with the following individuals added to the list of Contributing Authors:
 *
 *    Simon-Pierre Cadieux
 *    Eric S. Raymond
 *    Gilles Vollant
 *
 * and with the following additions to the disclaimer:
 *
 *    There is no warranty against interference with your enjoyment of the
 *    library or against infringement.  There is no warranty that our
 *    efforts or the library will fulfill any of your particular purposes
 *    or needs.  This library is provided with all faults, and the entire
 *    risk of satisfactory quality, performance, accuracy, and effort is with
 *    the user.
 *
 * libpng versions 0.97, January 1998, through 1.0.6, March 20, 2000, are
 * Copyright (c) 1998, 1999, 2000 Glenn Randers-Pehrson, and are
 * distributed according to the same disclaimer and license as libpng-0.96,
 * with the following individuals added to the list of Contributing Authors:
 *
 *    Tom Lane
 *    Glenn Randers-Pehrson
 *    Willem van Schaik
 *
 * libpng versions 0.89, June 1996, through 0.96, May 1997, are
 * Copyright (c) 1996, 1997 Andreas Dilger
 * Distributed according to the same disclaimer and license as libpng-0.88,
 * with the following individuals added to the list of Contributing Authors:
 *
 *    John Bowler
 *    Kevin Bracey
 *    Sam Bushell
 *    Magnus Holmgren
 *    Greg Roelofs
 *    Tom Tanner
 *
 * libpng versions 0.5, May 1995, through 0.88, January 1996, are
 * Copyright (c) 1995, 1996 Guy Eric Schalnat, Group 42, Inc.
 *
 * For the purposes of this copyright and license, "Contributing Authors"
 * is defined as the following set of individuals:
 *
 *    Andreas Dilger
 *    Dave Martindale
 *    Guy Eric Schalnat
 *    Paul Schmidt
 *    Tim Wegner
 *
 * The PNG Reference Library is supplied "AS IS".  The Contributing Authors
 * and Group 42, Inc. disclaim all warranties, expressed or implied,
 * including, without limitation, the warranties of merchantability and of
 * fitness for any purpose.  The Contributing Authors and Group 42, Inc.
 * assume no liability for direct, indirect, incidental, special, exemplary,
 * or consequential damages, which may result from the use of the PNG
 * Reference Library, even if advised of the possibility of such damage.
 *
 * Permission is hereby granted to use, copy, modify, and distribute this
 * source code, or portions hereof, for any purpose, without fee, subject
 * to the following restrictions:
 *
 *   1. The origin of this source code must not be misrepresented.
 *
 *   2. Altered versions must be plainly marked as such and must not
 *      be misrepresented as being the original source.
 *
 *   3. This Copyright notice may not be removed or altered from
 *      any source or altered source distribution.
 *
 * The Contributing Authors and Group 42, Inc. specifically permit, without
 * fee, and encourage the use of this source code as a component to
 * supporting the PNG file format in commercial products.  If you use this
 * source code in a product, acknowledgment is not required but would be
 * appreciated.
 *)

(*
 * A "png_get_copyright" function is available, for convenient use in "about"
 * boxes and the like:
 *
 *     printf("%s", png_get_copyright(NULL));
 *
 * Also, the PNG logo (in PNG format, of course) is supplied in the
 * files "pngbar.png" and "pngbar.jpg (88x31) and "pngnow.png" (98x31).
 *)

(*
 * Libpng is OSI Certified Open Source Software.  OSI Certified is a
 * certification mark of the Open Source Initiative.
 *)

(*
 * The contributing authors would like to thank all those who helped
 * with testing, bug fixes, and patience.  This wouldn't have been
 * possible without all of you.
 *
 * Thanks to Frank J. T. Wojcik for helping with the documentation.
 *)

(*
 * Y2K compliance in libpng:
 * =========================
 *
 *    March 29, 2012
 *
 *    Since the PNG Development group is an ad-hoc body, we can't make
 *    an official declaration.
 *
 *    This is your unofficial assurance that libpng from version 0.71 and
 *    upward through 1.5.10 are Y2K compliant.  It is my belief that
 *    earlier versions were also Y2K compliant.
 *
 *    Libpng only has two year fields.  One is a 2-byte unsigned integer
 *    that will hold years up to 65535.  The other holds the date in text
 *    format, and will hold years up to 9999.
 *
 *    The integer is
 *        "png_uint_16 year" in png_time_struct.
 *
 *    The string is
 *        "png_char time_buffer" in png_struct
 *
 *    There are seven time-related functions:
 *        png.c: png_convert_to_rfc_1123() in png.c
 *          (formerly png_convert_to_rfc_1152() in error)
 *        png_convert_from_struct_tm() in pngwrite.c, called in pngwrite.c
 *        png_convert_from_time_t() in pngwrite.c
 *        png_get_tIME() in pngget.c
 *        png_handle_tIME() in pngrutil.c, called in pngread.c
 *        png_set_tIME() in pngset.c
 *        png_write_tIME() in pngwutil.c, called in pngwrite.c
 *
 *    All handle dates properly in a Y2K environment.  The
 *    png_convert_from_time_t() function calls gmtime() to convert from system
 *    clock time, which returns (year - 1900), which we properly convert to
 *    the full 4-digit year.  There is a possibility that applications using
 *    libpng are not passing 4-digit years into the png_convert_to_rfc_1123()
 *    function, or that they are incorrectly passing only a 2-digit year
 *    instead of "year - 1900" into the png_convert_from_struct_tm() function,
 *    but this is not under our control.  The libpng documentation has always
 *    stated that it works with 4-digit years, and the APIs have been
 *    documented as such.
 *
 *    The tIME chunk itself is also Y2K compliant.  It uses a 2-byte unsigned
 *    integer to hold the year, and can hold years as large as 65535.
 *
 *    zlib, upon which libpng depends, is also Y2K compliant.  It contains
 *    no date-related code.
 *
 *       Glenn Randers-Pehrson
 *       libpng maintainer
 *       PNG Development Group
 *)

{$IFDEF FPC}
  {$MODE Delphi}

  {$IFDEF CPUI386}
    {$DEFINE CPU386}
    {$ASMMODE INTEL}
  {$ENDIF}

  {$IFNDEF WIN32}
    {$LINKLIB c}
  {$ENDIF}
{$ENDIF}

{$EXTENDEDSYNTAX ON}
{$ALIGN 8}
{$MINENUMSIZE 4}

interface

const
  {$IFDEF WIN32}
    LIB_PNG_NAME = 'libpng15.dll';
  {$ELSE}
    LIB_PNG_NAME = 'libpng15.so.0';
  {$ENDIF}

type
  png_byte =            byte;
  png_int_16 =          smallint;
  png_uint_16 =         word;
  png_int_32 =          integer;
  png_uint_32 =         cardinal;
  png_size_t =          cardinal;
  png_alloc_size_t =    png_size_t;
  png_char =            AnsiChar;
  
  (* Typedef for floating-point numbers that are converted
   * to fixed-point with a multiple of 100,000, e.g., gamma
   *)
  png_fixed_point =     png_int_32;  

  png_bytep =           ^png_byte;
  png_int_16p =         ^png_int_16;
  png_uint_16p =        ^png_uint_16;
  png_int_32p =         ^png_int_32;
  png_uint_32p =        ^png_uint_32;
  png_fixed_pointp =    ^png_fixed_point;
  png_charp =           pAnsiChar;
  png_const_charp =     pAnsiChar;
  png_voidp =           pointer;
  png_doublep =         ^double;

  png_bytepp =          ^png_bytep;
  png_int_16pp =        ^png_int_16p;
  png_uint_16pp =       ^png_uint_16p;
  png_int_32pp =        ^png_int_32p;
  png_uint_32pp =       ^png_uint_32p;
  png_fixed_pointpp =   ^png_fixed_pointp;
  png_charpp =          ^png_charp;
  png_voidpp =          ^png_voidp;

  png_charppp =         ^png_charpp;

  png_FILE_p =          cardinal;

  (* compatibility for C *)
  TM = record
    tm_sec: Integer;     (* Seconds *)
    tm_min: Integer;     (* Minutes *)
    tm_hour: Integer;    (* Hours *)
    tm_mday: Integer;    (* Day of the month *)
    tm_mon: Integer;     (* Months *)
    tm_year: Integer;    (* Years since 1900 *)
    tm_wday: Integer;    (* Days since Sunday (0 - 6) *)
    tm_yday: Integer;    (* Day of the year (0 - 365) *)
    tm_isdst: Integer;   (* Daylight savings time flag *)
  end;
  TMp = ^TM;

  time_t = Longint;  

const
(* This is not the place to learn how to use libpng. The file libpng-manual.txt
 * describes how to use libpng, and the file example.c summarizes it
 * with some code on which to build.  This file is useful for looking
 * at the actual function definitions and structure components.
 *
 * If you just need to read a PNG file and don't want to read the documentation
 * skip to the end of this file and read the section entitled 'simplified API'.
 *)

(* Version information for png.h - this should match the version in png.c *)
  PNG_LIBPNG_VER_STRING = '1.5.10';
  PNG_HEADER_VERSION_STRING = ' libpng version 1.5.10 - March 29, 2012'#13#10;

(* These should match the first 3 components of PNG_LIBPNG_VER_STRING: *)
  PNG_LIBPNG_VER_SONUM = 15;
  PNG_LIBPNG_VER_DLLNUM = 15;

(* These should match the first 3 components of PNG_LIBPNG_VER_STRING: *)
  PNG_LIBPNG_VER_MAJOR   =  1;
  PNG_LIBPNG_VER_MINOR   =  5;
  PNG_LIBPNG_VER_RELEASE = 10;

(* This should match the numeric part of the final component of
 * PNG_LIBPNG_VER_STRING, omitting any leading zero:
 *)

  PNG_LIBPNG_VER_BUILD = 0;

(* Release Status *)
  PNG_LIBPNG_BUILD_ALPHA    = 1;
  PNG_LIBPNG_BUILD_BETA     = 2;
  PNG_LIBPNG_BUILD_RC       = 3;
  PNG_LIBPNG_BUILD_STABLE   = 4;
  PNG_LIBPNG_BUILD_RELEASE_STATUS_MASK = 7;

(* Release-Specific Flags *)
  PNG_LIBPNG_BUILD_PATCH   = 8;  (* Can be OR'ed with
                                    PNG_LIBPNG_BUILD_STABLE only *)
  PNG_LIBPNG_BUILD_PRIVATE = 16; (* Cannot be OR'ed with
                                    PNG_LIBPNG_BUILD_SPECIAL *)
  PNG_LIBPNG_BUILD_SPECIAL = 32; (* Cannot be OR'ed with
                                    PNG_LIBPNG_BUILD_PRIVATE *)

  PNG_LIBPNG_BUILD_BASE_TYPE = PNG_LIBPNG_BUILD_STABLE;

(* Careful here.  At one time, Guy wanted to use 082, but that would be octal.
 * We must not include leading zeros.
 * Versions 0.7 through 1.0.0 were in the range 0 to 100 here (only
 * version 1.0.0 was mis-numbered 100 instead of 10000).  From
 * version 1.0.1 it's    xxyyzz, where x=major, y=minor, z=release
 *)
  PNG_LIBPNG_VER 10510 (* 1.5.10 *)

type
(* Three color definitions.  The order of the red, green, and blue, (and the
  exact size) is not important, although the size of the fields need to
  be png_byte or png_uint_16 (as defined below). *)

  png_color = record 
    red: png_byte;
    green: png_byte;
    blue: png_byte;
  end;
  png_colorp = ^png_color;
  png_const_colorp = ^png_color;
  png_colorpp = ^png_colorp;

  png_color_16 = record
    index: png_byte;       (* used for palette files *)
    red: png_uint_16;      (* for use in red green blue files *)
    green: png_uint_16;
    blue: png_uint_16;
    gray: png_uint_16;     (* for use in grayscale files *)
  end;
  png_color_16p = ^png_color_16;
  png_const_color_16p = ^png_color_16;
  png_color_16pp = ^png_color_16p;

  png_color_8 = record
    red: png_byte;      (* for use in red green blue files *)
    green: png_byte;
    blue: png_byte;
    gray: png_byte;     (* for use in grayscale files *)
    alpha: png_byte;    (* for alpha channel files *)
  end;
  png_color_8p = ^png_color_8;
  png_const_color_8p = ^png_color_8;
  png_color_8pp = ^png_color_8p;

(* The following two structures are used for the in-core representation
  of sPLT chunks. *)

  png_sPLT_entry = record
    red: png_uint_16;
    green: png_uint_16;
    blue: png_uint_16;
    alpha: png_uint_16;
    frequency: png_uint_16;
  end;
  png_sPLT_entryp = ^png_sPLT_entry;
  png_const_sPLT_entryp = ^png_sPLT_entry;
  png_sPLT_entrypp = ^png_sPLT_entryp;

(* When the depth of the sPLT palette is 8 bits, the color and alpha samples
  occupy the LSB of their respective members, and the MSB of each member
  is zero-filled.  The frequency member always occupies the full 16 bits. *)

  png_sPLT_t = record
    name: png_charp;            (* palette name *)
    depth: png_byte;            (* depth of palette samples *)
    entries: png_sPLT_entryp;   (* palette entries *)
    nentries: png_int_32;       (* number of palette entries *)
  end;
  png_sPLT_tp = ^png_sPLT_t;
  png_const_sPLT_tp = ^png_sPLT_t;
  png_sPLT_tpp = ^png_sPLT_tp;

{.$ifdef PNG_TEXT_SUPPORTED}
(* png_text holds the contents of a text/ztxt/itxt chunk in a PNG file,
  and whether that contents is compressed or not.  The "key" field
  points to a regular zero-terminated C string.  The "text", "lang", and
  "lang_key" fields can be regular C strings, empty strings, or NULL pointers.
  However, the * structure returned by png_get_text() will always contain
  regular zero-terminated C strings (possibly empty), never NULL pointers,
  so they can be safely used in printf() and other string-handling functions. *)

  png_text = record
    compression: Integer;    (* compression value:
                               -1: tEXt, none
                                0: zTXt, deflate
                                1: iTXt, none
                                2: iTXt, deflate *)
    key: png_charp;          (* keyword, 1-79 character description of "text" *)
    text: png_charp;         (* comment, may be an empty string (ie "")
                                or a NULL pointer *)
    text_length: png_size_t; (* length of the text string *)

    itxt_length: png_size_t; (* length of the itxt string *)
    lang: png_charp;         (* language code, 0-79 characters
                                or a NULL pointer *)
    lang_key: png_charp;     (* keyword translated UTF-8 string, 0 or more
                                chars or a NULL pointer *)
  end;
  png_textp = ^png_text;
  png_const_textp = ^png_text;
  png_textpp = ^png_textp;
{.$endif}

const
(* Supported compression types for text in PNG files (tEXt, and zTXt).
  The values of the PNG_TEXT_COMPRESSION_ defines should NOT be changed. *)
  PNG_TEXT_COMPRESSION_NONE_WR = -3;
  PNG_TEXT_COMPRESSION_zTXt_WR = -2;
  PNG_TEXT_COMPRESSION_NONE    = -1;
  PNG_TEXT_COMPRESSION_zTXt    =  0;
  PNG_ITXT_COMPRESSION_NONE    =  1;
  PNG_ITXT_COMPRESSION_zTXt    =  2;
  PNG_TEXT_COMPRESSION_LAST    =  3;  (* Not a valid value *)

type
(* png_time is a way to hold the time in an machine independent way.
  Two conversions are provided, both from time_t and struct tm.  There
  is no portable way to convert to either of these structures, as far
  as I know.  If you know of a portable way, send it to me.  As a side
  note - PNG has always been Year 2000 compliant! *)

  png_time = record
    year: png_uint_16; (* full year, as in, 1995 *)
    month: png_byte;   (* month of year, 1 - 12 *)
    day: png_byte;     (* day of month, 1 - 31 *)
    hour: png_byte;    (* hour of day, 0 - 23 *)
    minute: png_byte;  (* minute of hour, 0 - 59 *)
    second: png_byte;  (* second of minute, 0 - 60 (for leap seconds) *)
  end;
  png_timep = ^png_time;
  png_const_timep = ^png_time;
  png_timepp = ^png_timep;

{.$if defined(PNG_UNKNOWN_CHUNKS_SUPPORTED) or
    defined(PNG_HANDLE_AS_UNKNOWN_SUPPORTED)}
(* png_unknown_chunk is a structure to hold queued chunks for which there is
  no specific support.  The idea is that we can use this to queue
  up private chunks for output even though the library doesn't actually
  know about their semantics. *)
  
  png_unknown_chunk = record
    name: array [0..4] of png_byte;
    data: ^png_byte;
    size: png_size_t;

    (* libpng-using applications should NOT directly modify this byte. *)
    location: png_byte; (* mode of operation at read time *)
  end;
  png_unknown_chunkp = ^png_unknown_chunk;
  png_const_unknown_chunkp = ^png_unknown_chunk;
  png_unknown_chunkpp = ^png_unknown_chunkp;
{.$endif}

(* Values for the unknown chunk location byte *)
const
  PNG_HAVE_IHDR  = $01;
  PNG_HAVE_PLTE  = $02;
  PNG_AFTER_IDAT = $08;

type
(* The complete definition of png_info has, as of libpng-1.5.0,
 * been moved into a separate header file that is not accessible to
 * applications.  Read libpng-manual.txt or libpng.3 for more info.
 *)
  png_infop = ^png_info;
  png_const_infop = ^png_info;
  png_infopp = ^png_infop;

// --- from pnginfo.h ---
(* png_info is a structure that holds the information in a PNG file so
  that the application can find out the characteristics of the image.
  If you are reading the file, this structure will tell you what is
  in the PNG file.  If you are writing the file, fill in the information
  you want to put into the PNG file, then call png_write_info().
  The names chosen should be very close to the PNG specification, so
  consult that document for information about the meaning of each field.
 
  With libpng < 0.95, it was only possible to directly set and read the
  the values in the png_info_struct, which meant that the contents and
  order of the values had to remain fixed.  With libpng 0.95 and later,
  however, there are now functions that abstract the contents of
  png_info_struct from the application, so this makes it easier to use
  libpng with dynamic libraries, and even makes it possible to use
  libraries that don't have all of the libpng ancillary chunk-handing
  functionality.
 
  In any case, the order of the parameters in png_info_struct should NOT
  be changed for as long as possible to keep compatibility with applications
  that use the old direct-access method with png_info_struct.

  The following members may have allocated storage attached that should be
  cleaned up before the structure is discarded: palette, trans, text,
  pcal_purpose, pcal_units, pcal_params, hist, iccp_name, iccp_profile,
  splt_palettes, scal_unit, row_pointers, and unknowns.   By default, these
  are automatically freed when the info structure is deallocated, if they were
  allocated internally by libpng.  This behavior can be changed by means
  of the png_data_freer() function.

  More allocation details: all the chunk-reading functions that
  change these members go through the corresponding png_set_*
  functions.  A function to clear these members is available: see
  png_free_data().  The png_set_* functions do not depend on being
  able to point info structure members to any of the storage they are
  passed (they make their own copies), EXCEPT that the png_set_text
  functions use the same storage passed to them in the text_ptr or
  itxt_ptr structure argument, and the png_set_rows and png_set_unknowns
  functions do not make their own copies.*)

  png_info = record
    (* the following are necessary for every PNG file *)
    width: png_uint_32;       (* width of image in pixels (from IHDR) *)
    height: png_uint_32;      (* height of image in pixels (from IHDR) *)
    valid: png_uint_32;       (* valid chunk data (see PNG_INFO_ below) *)
    rowbytes: png_size_t;     (* bytes needed to hold an untransformed row *)
    palette: png_colorp;      (* array of color values (valid & PNG_INFO_PLTE) *)
    num_palette: png_uint_16; (* number of color entries in "palette" (PLTE) *)
    num_trans: png_uint_16;   (* number of transparent palette color (tRNS) *)
    bit_depth: png_byte;      (* 1, 2, 4, 8, or 16 bits/channel (from IHDR) *)
    color_type: png_byte;     (* see PNG_COLOR_TYPE_ below (from IHDR) *)
    (* The following three should have been named *_method not *_type *)
    compression_type: png_byte; (* must be PNG_COMPRESSION_TYPE_BASE (IHDR) *)
    filter_type: png_byte;    (* must be PNG_FILTER_TYPE_BASE (from IHDR) *)
    interlace_type: png_byte; (* One of PNG_INTERLACE_NONE, PNG_INTERLACE_ADAM7 *)

    (* The following is informational only on read, and not used on writes. *)
    channels: png_byte;       (* number of data channels per pixel (1, 2, 3, 4) *)
    pixel_depth: png_byte;    (* number of bits per pixel *)
    spare_byte: png_byte;     (* to align the data, and for future use *)
    signature: array[0..7] of png_byte; (* magic bytes read by libpng from start of file *)

    (* The rest of the data is optional.  If you are reading, check the
      valid field to see if the information in these are valid.  If you
      are writing, set the valid field to those chunks you want written,
      and initialize the appropriate fields below. *)

{.$if defined(PNG_gAMA_SUPPORTED)}
    (* The gAMA chunk describes the gamma characteristics of the system
      on which the image was created, normally in the range [1.0, 2.5].
      Data is valid if (valid & PNG_INFO_gAMA) is non-zero. *)
    gamma: png_fixed_point (* gamma value of image, if (valid & PNG_INFO_gAMA) *)
{.$endif}

{.$ifdef PNG_sRGB_SUPPORTED}
    (* GR-P, 0.96a *)
    (* Data valid if (valid & PNG_INFO_sRGB) non-zero. *)
    srgb_intent: png_byte;  (* sRGB rendering intent [0, 1, 2, or 3] *)
{.$endif}

{.$ifdef PNG_TEXT_SUPPORTED}
    (* The tEXt, and zTXt chunks contain human-readable textual data in
      uncompressed, compressed, and optionally compressed forms, respectively.
      The data in "text" is an array of pointers to uncompressed,
      null-terminated C strings. Each chunk has a keyword that describes the
      textual data contained in that chunk.  Keywords are not required to be
      unique, and the text string may be empty.  Any number of text chunks may
      be in an image. *)
    num_text: Integer;  (* number of comments read/to write *)
    max_text: integer;  (* current size of text array *)
    text: png_textp;  (* array of comments read/to write *)
{.$endif} (* PNG_TEXT_SUPPORTED *)

{.$ifdef PNG_tIME_SUPPORTED}
    (* The tIME chunk holds the last time the displayed image data was
      modified.  See the png_time struct for the contents of this struct. *)
    mod_time: png_time;
{.$endif}

{.$ifdef PNG_sBIT_SUPPORTED}
    (* The sBIT chunk specifies the number of significant high-order bits
      in the pixel data.  Values are in the range [1, bit_depth], and are
      only specified for the channels in the pixel data.  The contents of
      the low-order bits is not specified.  Data is valid if
      (valid & PNG_INFO_sBIT) is non-zero. *)
    sig_bit: png_color_8;  (* significant bits in color channels *)
{.$endif}

{.$if defined(PNG_tRNS_SUPPORTED) or defined(PNG_READ_EXPAND_SUPPORTED) or
      defined(PNG_READ_BACKGROUND_SUPPORTED)}
    (* The tRNS chunk supplies transparency data for paletted images and
      other image types that don't need a full alpha channel.  There are
      "num_trans" transparency values for a paletted image, stored in the
      same order as the palette colors, starting from index 0.  Values
      for the data are in the range [0, 255], ranging from fully transparent
      to fully opaque, respectively.  For non-paletted images, there is a
      single color specified that should be treated as fully transparent.
      Data is valid if (valid & PNG_INFO_tRNS) is non-zero. *)
    trans_alpha: png_bytep;  (* transparent values for paletted image *)
    trans_color: png_color_16;  (* transparent color for non-palette image *)
{.$endif}

{.$if defined(PNG_bKGD_SUPPORTED) or defined(PNG_READ_BACKGROUND_SUPPORTED)}
    (* The bKGD chunk gives the suggested image background color if the
      display program does not have its own background color and the image
      is needs to composited onto a background before display.  The colors
      in "background" are normally in the same color space/depth as the
      pixel data.  Data is valid if (valid & PNG_INFO_bKGD) is non-zero. *)
    background: png_color_16;
{.$endif}

{.$ifdef PNG_oFFs_SUPPORTED}
    (* The oFFs chunk gives the offset in "offset_unit_type" units rightwards
      and downwards from the top-left corner of the display, page, or other
      application-specific co-ordinate space.  See the PNG_OFFSET_ defines
      below for the unit types.  Valid if (valid & PNG_INFO_oFFs) non-zero. *)
    x_offset: png_int_32;  (* x offset on page *)
    y_offset: png_int_32;  (* y offset on page *)
    offset_unit_type: png_byte;  (* offset units type *)
{.$endif}

{.$ifdef PNG_pHYs_SUPPORTED}
    (* The pHYs chunk gives the physical pixel density of the image for
      display or printing in "phys_unit_type" units (see PNG_RESOLUTION_
      defines below).  Data is valid if (valid & PNG_INFO_pHYs) is non-zero. *)
    x_pixels_per_unit: png_uint_32;  (* horizontal pixel density *)
    y_pixels_per_unit: png_uint_32;  (* vertical pixel density *)
    phys_unit_type: png_byte;  (* resolution type (see PNG_RESOLUTION_ below) *)
{.$endif}

{.$ifdef PNG_hIST_SUPPORTED}
    (* The hIST chunk contains the relative frequency or importance of the
      various palette entries, so that a viewer can intelligently select a
      reduced-color palette, if required.  Data is an array of "num_palette"
      values in the range [0,65535]. Data valid if (valid & PNG_INFO_hIST)
      is non-zero. *)
    hist: png_uint_16p;
{.$endif}

{.$ifdef PNG_cHRM_SUPPORTED}
    (* The cHRM chunk describes the CIE color characteristics of the monitor
      on which the PNG was created.  This data allows the viewer to do gamut
      mapping of the input image to ensure that the viewer sees the same
      colors in the image as the creator.  Values are in the range
      [0.0, 0.8].  Data valid if (valid & PNG_INFO_cHRM) non-zero. *)
    x_white: png_fixed_point;
    y_white: png_fixed_point;
    x_red: png_fixed_point;
    y_red: png_fixed_point;
    x_green: png_fixed_point;
    y_green: png_fixed_point;
    x_blue: png_fixed_point;
    y_blue: png_fixed_point;
{.$endif}

{.$ifdef PNG_pCAL_SUPPORTED}
    (* The pCAL chunk describes a transformation between the stored pixel
      values and original physical data values used to create the image.
      The integer range [0, 2^bit_depth - 1] maps to the floating-point
      range given by [pcal_X0, pcal_X1], and are further transformed by a
      (possibly non-linear) transformation function given by "pcal_type"
      and "pcal_params" into "pcal_units".  Please see the PNG_EQUATION_
      defines below, and the PNG-Group's PNG extensions document for a
      complete description of the transformations and how they should be
      implemented, and for a description of the ASCII parameter strings.
      Data values are valid if (valid & PNG_INFO_pCAL) non-zero. *)
    pcal_purpose: png_charp;  (* pCAL chunk description string *)
    pcal_X0: png_int_32;      (* minimum value *)
    pcal_X1: png_int_32;      (* maximum value *)
    pcal_units: png_charp;    (* Latin-1 string giving physical units *)
    pcal_params: png_charpp;  (* ASCII strings containing parameter values *)
    pcal_type: png_byte;      (* equation type (see PNG_EQUATION_ below) *)
    pcal_nparams: png_byte;   (* number of parameters given in pcal_params *)
{.$endif}

    (* New members added in libpng-1.0.6 *)
    free_me: png_uint_32;     (* flags items libpng is responsible for freeing *)

{.$if defined(PNG_UNKNOWN_CHUNKS_SUPPORTED) or defined(PNG_HANDLE_AS_UNKNOWN_SUPPORTED)}
    (* Storage for unknown chunks that the library doesn't recognize. *)
    unknown_chunks: png_unknown_chunkp;
    unknown_chunks_num: Integer;
{.$endif}

{.$ifdef PNG_iCCP_SUPPORTED}
    (* iCCP chunk data. *)
    iccp_name: png_charp;     (* profile name *)
    iccp_profile: png_bytep;  (* International Color Consortium profile data *)
    iccp_proflen: png_uint_32;  (* ICC profile data length *)
    iccp_compression: png_byte; (* Always zero *)
{.$endif}

{.$ifdef PNG_sPLT_SUPPORTED}
    (* data on sPLT chunks (there may be more than one). *)
    splt_palettes: png_sPLT_tp;
    splt_palettes_num: png_uint_32;
{.$endif}

{.$ifdef PNG_sCAL_SUPPORTED}
    (* The sCAL chunk describes the actual physical dimensions of the
      subject matter of the graphic.  The chunk contains a unit specification
      a byte value, and two ASCII strings representing floating-point
      values.  The values are width and height corresponsing to one pixel
      in the image.  This external representation is converted to double
      here.  Data values are valid if (valid & PNG_INFO_sCAL) is non-zero. *)
    scal_unit: png_byte;         (* unit of physical scale *)
    scal_s_width: png_charp;     (* string containing height *)
    scal_s_height: png_charp;    (* string containing width *)
{.$endif}

{.$ifdef PNG_INFO_IMAGE_SUPPORTED}
    (* Memory has been allocated if (valid & PNG_ALLOCATED_INFO_ROWS) non-zero *)
    (* Data valid if (valid & PNG_INFO_IDAT) non-zero *)
    row_pointers: png_bytepp;        (* the image bits *)
{.$endif}
  end;
// --- pnhinfo.h finish hire ---

const
  (* Maximum positive integer used in PNG is (2^31)-1 *)
  PNG_UINT_31_MAX = png_uint_32($7FFFFFFF);
  PNG_UINT_32_MAX = png_uint_32(-1);
  PNG_SIZE_MAX = png_size_t(-1);
  
  (* These are constants for fixed point values encoded in the
   * PNG specification manner (x100000)
   *)
  PNG_FP_1    = 100000;
  PNG_FP_HALF = 50000;
  PNG_FP_MAX  = png_fixed_point($7FFFFFFF);
  PNG_FP_MIN  = (-PNG_FP_MAX);
  
  (* These describe the color_type field in png_info. *)
  (* color type masks *)
  PNG_COLOR_MASK_PALETTE    = 1;
  PNG_COLOR_MASK_COLOR      = 2;
  PNG_COLOR_MASK_ALPHA      = 4;

  (* color types.  Note that not all combinations are legal *)
  PNG_COLOR_TYPE_GRAY       = 0;
  PNG_COLOR_TYPE_PALETTE    = PNG_COLOR_MASK_COLOR or PNG_COLOR_MASK_PALETTE;
  PNG_COLOR_TYPE_RGB        = PNG_COLOR_MASK_COLOR;
  PNG_COLOR_TYPE_RGB_ALPHA  = PNG_COLOR_MASK_COLOR or PNG_COLOR_MASK_ALPHA;
  PNG_COLOR_TYPE_GRAY_ALPHA = PNG_COLOR_MASK_ALPHA;
  (* aliases *)
  PNG_COLOR_TYPE_RGBA       = PNG_COLOR_TYPE_RGB_ALPHA;
  PNG_COLOR_TYPE_GA         = PNG_COLOR_TYPE_GRAY_ALPHA;

  (* This is for compression type. PNG 1.0-1.2 only define the single type. *)
  PNG_COMPRESSION_TYPE_BASE     = 0;  (* Deflate method 8, 32K window *)
  PNG_COMPRESSION_TYPE_DEFAULT  = PNG_COMPRESSION_TYPE_BASE;

  (* This is for filter type. PNG 1.0-1.2 only define the single type. *)
  PNG_FILTER_TYPE_BASE        =  0;  (* Single row per-byte filtering *)
  PNG_INTRAPIXEL_DIFFERENCING = 64;  (* Used only in MNG datastreams *)
  PNG_FILTER_TYPE_DEFAULT     = PNG_FILTER_TYPE_BASE;

  (* These are for the interlacing type.  These values should NOT be changed. *)
  PNG_INTERLACE_NONE        = 0;  (* Non-interlaced image *)
  PNG_INTERLACE_ADAM7       = 1;  (* Adam7 interlacing *)
  PNG_INTERLACE_LAST        = 2;  (* Not a valid value *)

  (* These are for the oFFs chunk.  These values should NOT be changed. *)
  PNG_OFFSET_PIXEL          = 0;  (* Offset in pixels *)
  PNG_OFFSET_MICROMETER     = 1;  (* Offset in micrometers (1/10^6 meter) *)
  PNG_OFFSET_LAST           = 2;  (* Not a valid value *)

  (* These are for the pCAL chunk.  These values should NOT be changed. *)
  PNG_EQUATION_LINEAR       = 0;  (* Linear transformation *)
  PNG_EQUATION_BASE_E       = 1;  (* Exponential base e transform *)
  PNG_EQUATION_ARBITRARY    = 2;  (* Arbitrary base exponential transform *)
  PNG_EQUATION_HYPERBOLIC   = 3;  (* Hyperbolic sine transformation *)
  PNG_EQUATION_LAST         = 4;  (* Not a valid value *)

  (* These are for the sCAL chunk.  These values should NOT be changed. *)
  PNG_SCALE_UNKNOWN         = 0;  (* unknown unit (image scale) *)
  PNG_SCALE_METER           = 1;  (* meters per pixel *)
  PNG_SCALE_RADIAN          = 2;  (* radians per pixel *)
  PNG_SCALE_LAST            = 3;  (* Not a valid value *)

  (* These are for the pHYs chunk.  These values should NOT be changed. *)
  PNG_RESOLUTION_UNKNOWN    = 0;  (* pixels/unknown unit (aspect ratio) *)
  PNG_RESOLUTION_METER      = 1;  (* pixels/meter *)
  PNG_RESOLUTION_LAST       = 2;  (* Not a valid value *)

  (* These are for the sRGB chunk.  These values should NOT be changed. *)
  PNG_sRGB_INTENT_PERCEPTUAL = 0;
  PNG_sRGB_INTENT_RELATIVE   = 1;
  PNG_sRGB_INTENT_SATURATION = 2;
  PNG_sRGB_INTENT_ABSOLUTE   = 3;
  PNG_sRGB_INTENT_LAST       = 4;  (* Not a valid value *)

  (* This is for text chunks *)
  PNG_KEYWORD_MAX_LENGTH     = 79;

  (* Maximum number of entries in PLTE/sPLT/tRNS arrays *)
  PNG_MAX_PALETTE_LENGTH     = 256;

  (* These determine if an ancillary chunk's data has been successfully read
    from the PNG header, or if the application has filled in the corresponding
    data in the info_struct to be written into the output file.  The values
    of the PNG_INFO_<chunk> defines should NOT be changed. *)
  PNG_INFO_gAMA = $0001;
  PNG_INFO_sBIT = $0002;
  PNG_INFO_cHRM = $0004;
  PNG_INFO_PLTE = $0008;
  PNG_INFO_tRNS = $0010;
  PNG_INFO_bKGD = $0020;
  PNG_INFO_hIST = $0040;
  PNG_INFO_pHYs = $0080;
  PNG_INFO_oFFs = $0100;
  PNG_INFO_tIME = $0200;
  PNG_INFO_pCAL = $0400;
  PNG_INFO_sRGB = $0800;  (* GR-P, 0.96a *)
  PNG_INFO_iCCP = $1000;  (* ESR, 1.0.6 *)
  PNG_INFO_sPLT = $2000;  (* ESR, 1.0.6 *)
  PNG_INFO_sCAL = $4000;  (* ESR, 1.0.6 *)
  PNG_INFO_IDAT = $8000;  (* ESR, 1.0.6 *)

type
(* This is used for the transformation routines, as some of them
  change these values for the row.  It also should enable using
  the routines for other purposes. *)
  png_row_info = record
    width: png_uint_32;  (* width of row *)
    rowbytes: png_size_t;  (* number of bytes in row *)
    color_type: png_byte;  (* color type of row *)
    bit_depth: png_byte;  (* bit depth of row *)
    channels: png_byte;  (* number of channels (1, 2, 3, or 4) *)
    pixel_depth: png_byte;  (* bits per pixel (depth * channels) *)
  end;
  png_row_infop = ^png_row_info;
  png_row_infopp = ^png_row_infop;
  
const
(* Transform masks for the high-level interface *)
  PNG_TRANSFORM_IDENTITY       = $0000;    (* read and write *)
  PNG_TRANSFORM_STRIP_16       = $0001;    (* read only *)
  PNG_TRANSFORM_STRIP_ALPHA    = $0002;    (* read only *)
  PNG_TRANSFORM_PACKING        = $0004;    (* read and write *)
  PNG_TRANSFORM_PACKSWAP       = $0008;    (* read and write *)
  PNG_TRANSFORM_EXPAND         = $0010;    (* read only *)
  PNG_TRANSFORM_INVERT_MONO    = $0020;    (* read and write *)
  PNG_TRANSFORM_SHIFT          = $0040;    (* read and write *)
  PNG_TRANSFORM_BGR            = $0080;    (* read and write *)
  PNG_TRANSFORM_SWAP_ALPHA     = $0100;    (* read and write *)
  PNG_TRANSFORM_SWAP_ENDIAN    = $0200;    (* read and write *)
  PNG_TRANSFORM_INVERT_ALPHA   = $0400;    (* read and write *)
  PNG_TRANSFORM_STRIP_FILLER   = $0800;    (* WRITE only *)
  (* Added to libpng-1.2.34 *)
  PNG_TRANSFORM_STRIP_FILLER_BEFORE = PNG_TRANSFORM_STRIP_FILLER;
  PNG_TRANSFORM_STRIP_FILLER_AFTER = $1000; (* write only *)
  (* Added to libpng-1.4.0 *)
  PNG_TRANSFORM_GRAY_TO_RGB    = $2000;    (* read only *)
  (* Added to libpng-1.5.4 *)
  PNG_TRANSFORM_EXPAND_16      = $4000;    (* read only *)
  PNG_TRANSFORM_SCALE_16       = $8000;    (* read only *)
  
(* Flags for MNG supported features *)
  PNG_FLAG_MNG_EMPTY_PLTE     = $01;
  PNG_FLAG_MNG_FILTER_64      = $04;
  PNG_ALL_MNG_FEATURES        = $05;

// ---- from pngstruct.h ----  
  PNG_ZLIB_UNINITIALIZED = 0
  PNG_ZLIB_FOR_IDAT      = 1
  PNG_ZLIB_FOR_TEXT      = 2 (* anything other than IDAT *)
  PNG_ZLIB_USE_MASK      = 3 (* bottom two bits *)
  PNG_ZLIB_IN_USE        = 4 (* a flag value *)
// --- end from pngstruct.h ----

type
(* These are the function types for the I/O functions and for the functions
 * that allow the user to override the default I/O functions with his or her
 * own.  The png_error_ptr type should match that of user-supplied warning
 * and error functions, while the png_rw_ptr type should match that of the
 * user read/write data functions.  Note that the 'write' function must not
 * modify the buffer it is passed. The 'read' function, on the other hand, is
 * expected to return the read data in the buffer.
 *)

  png_structp = ^png_struct;

  png_error_ptr = procedure(p1: png_structp; p2: png_const_charp); cdecl;
  png_rw_ptr = procedure(p1: png_structp; p2: png_bytep; p3: png_size_t); cdecl;
  png_flush_ptr = procedure(p1: png_structp); cdecl;
  png_read_status_ptr = procedure(p1: png_structp; p2: png_uint_32; p3: integer); cdecl;
  png_write_status_ptr = procedure(p1: png_structp; p2: png_uint_32; p3: integer); cdecl;

{.$ifdef PNG_PROGRESSIVE_READ_SUPPORTED}
  png_progressive_info_ptr = procedure(p1: png_structp; p2: png_infop); cdecl;
  png_progressive_end_ptr = procedure(p1: png_structp; p2: png_infop); cdecl;
  
  (* The following callback receives png_uint_32 row_number, int pass for the
   * png_bytep data of the row.  When transforming an interlaced image the
   * row number is the row number within the sub-image of the interlace pass, so
   * the value will increase to the height of the sub-image (not the full image)
   * then reset to 0 for the next pass.
   *
   * Use PNG_ROW_FROM_PASS_ROW(row, pass) and PNG_COL_FROM_PASS_COL(col, pass) to
   * find the output pixel (x,y) given an interlaced sub-image pixel
   * (row,col,pass).  (See below for these macros.)
   *)  
  png_progressive_row_ptr = procedure(p1: png_structp; p2: png_bytep; p3: png_uint_32; p4: integer); cdecl;
{.$endif}

{.$if defined(PNG_READ_USER_TRANSFORM_SUPPORTED) or 
  defined(PNG_WRITE_USER_TRANSFORM_SUPPORTED)}
  png_user_transform_ptr = procedure(p1: png_structp; p2: png_row_infop; p3: png_bytep); cdecl;
{.$endif}

{.$ifdef PNG_USER_CHUNKS_SUPPORTED}
  png_user_chunk_ptr = function(p1: png_structp; p2: png_unknown_chunkp): integer; cdecl;
{.$endif}
{.$ifdef PNG_UNKNOWN_CHUNKS_SUPPORTED}
  png_unknown_chunk_ptr = procedure(p1: png_structp); cdecl;
{.$endif}

{.$ifdef PNG_SETJMP_SUPPORTED}
(* This must match the function definition in <setjmp.h>, and the application
 * must include this before png.h to obtain the definition of jmp_buf.  The
 * function is required to be PNG_NORETURN, but this is not checked.  If the
 * function does return the application will crash via an abort() or similar
 * system level call.
 *
 * If you get a warning here while building the library you may need to make
 * changes to ensure that pnglibconf.h records the calling convention used by
 * your compiler.  This may be very difficult - try using a different compiler
 * to build the library!
 *)
//PNG_FUNCTION(void, (PNGCAPI *png_longjmp_ptr), PNGARG((jmp_buf, int)), typedef);
  png_longjmp_ptr = pointer;
{.$endif}

(* NOTE: prior to 1.5 these functions had no 'API' style declaration,
 * this allowed the zlib default functions to be used on Windows
 * platforms.  In 1.5 the zlib default malloc (which just calls malloc and
 * ignores the first argument) should be completely compatible with the
 * following.
 *)
  png_malloc_ptr = function(p1: png_structp; p2: png_alloc_size_t): png_voidp; cdecl;
  png_free_ptr = procedure(p1: png_structp; p2: png_voidp); cdecl;

// ---- from pngstruct.h ----
(* The structure that holds the information to read and write PNG files.
 * The only people who need to care about what is inside of this are the
 * people who will be modifying the library for their own special needs.
 * It should NOT be accessed directly by an application.
 *)
  
  png_struct = record
{.$ifdef PNG_SETJMP_SUPPORTED}  
    jmp_buf: array[0..15] of Integer; (* used in png_error *)
    longjmp_fn: png_longjmp_ptr; (* setjmp non-local goto function. *)
{.$endif}
    error_fn: png_error_ptr;    (* function for printing errors and aborting *)
{.$ifdef PNG_WARNINGS_SUPPORTED}
    warning_fn: png_error_ptr;  (* function for printing warnings *)
{.$endif} 
    error_ptr: png_voidp;       (* user supplied struct for error functions *)
    write_data_fn: png_rw_ptr;  (* function for writing output data *)
    read_data_fn: png_rw_ptr;   (* function for reading input data *)
    io_ptr: png_voidp;          (* ptr to application struct for I/O functions *)

{.$ifdef PNG_READ_USER_TRANSFORM_SUPPORTED}
    read_user_transform_fn: png_user_transform_ptr; (* user read transform *)
{.$endif}

{.$ifdef PNG_WRITE_USER_TRANSFORM_SUPPORTED}
    write_user_transform_fn: png_user_transform_ptr; (* user write transform *)
{.$endif

(* These were added in libpng-1.0.2 *)
{.$ifdef PNG_USER_TRANSFORM_PTR_SUPPORTED}
{.$if defined(PNG_READ_USER_TRANSFORM_SUPPORTED) or
      defined(PNG_WRITE_USER_TRANSFORM_SUPPORTED)}
    user_transform_ptr: png_voidp; (* user supplied struct for user transform *)
    user_transform_depth: png_byte;    (* bit depth of user transformed pixels *)
    user_transform_channels: png_byte; (* channels in user transformed pixels *)
{.$endif}
{.$endif}

    mode: png_uint_32;          (* tells us where we are in the PNG file *)
    flags: png_uint_32;         (* flags indicating various things to libpng *)
    transformations: png_uint_32; (* which transformations to perform *)

    zstream: pointer; //zstream;(* pointer to decompression structure (below) *)
    zbuf: png_bytep;            (* buffer for zlib *)
    zbuf_size: png_size_t; //uInt;(* size of zbuf *)
{.$ifdef PNG_WRITE_SUPPORTED}
(* Added in 1.5.4: state to keep track of whether the zstream has been
 * initialized and if so whether it is for IDAT or some other chunk.
 *)  
    zlib_state: png_uint_32;     (* State of zlib initialization *)
(* End of material added at libpng 1.5.4 *)

    zlib_level: Integer;        (* holds zlib compression level *)
    zlib_method: Integer;       (* holds zlib compression method *)
    zlib_window_bits: Integer;  (* holds zlib compression window bits *)
    zlib_mem_level: Integer;    (* holds zlib compression memory level *)
    zlib_strategy: Integer;     (* holds zlib compression strategy *)
{.$endif}
(* Added at libpng 1.5.4 *)
{.$if defined(PNG_WRITE_COMPRESSED_TEXT_SUPPORTED) or
      defined(PNG_WRITE_CUSTOMIZE_ZTXT_COMPRESSION_SUPPORTED)}
   zlib_text_level: Integer;            (* holds zlib compression level *)
   zlib_text_method: Integer;           (* holds zlib compression method *)
   zlib_text_window_bits: Integer;      (* holds zlib compression window bits *)
   zlib_text_mem_level: Integer;        (* holds zlib compression memory level *)
   zlib_text_strategy: Integer;         (* holds zlib compression strategy *)
{.$endif}
(* End of material added at libpng 1.5.4 *)

    width: png_uint_32;         (* width of image in pixels *)
    height: png_uint_32;        (* height of image in pixels *)
    num_rows: png_uint_32;      (* number of rows in current pass *)
    usr_width: png_uint_32;     (* width of row at start of write *)
    rowbytes: png_size_t;       (* size of row in bytes *)
    iwidth: png_uint_32;        (* width of current interlaced row in pixels *)
    row_number: png_uint_32;    (* current row in interlace pass *)
    chunk_name: png_uint_32;    (* PNG_CHUNK() id of current chunk *)
    prev_row: png_bytep;        (* buffer to save previous (unfiltered) row.
                                 * This is a pointer into big_prev_row
                                 *)    
    row_buf: png_bytep;         (* buffer to save current (unfiltered) row.
                                 * This is a pointer into big_row_buf
                                 *)
    sub_row: png_bytep;         (* buffer to save "sub" row when filtering *)
    up_row: png_bytep;          (* buffer to save "up" row when filtering *)
    avg_row: png_bytep;         (* buffer to save "avg" row when filtering *)
    paeth_row: png_bytep;       (* buffer to save "Paeth" row when filtering *)
    info_rowbytes: png_size_t;  (* Added in 1.5.4: cache of updated row bytes *)
  
    idat_size: png_uint_32;     (* current IDAT size for read *)
    crc: png_uint_32;           (* current chunk CRC value *)
    palette: png_colorp;        (* palette from the input file *)
    num_palette: png_uint_16;   (* number of color entries in palette *)
    
(* Added at libpng-1.5.10 *)
{.$ifdef PNG_CHECK_FOR_INVALID_INDEX_SUPPORTED}
   num_palette_max: Integer;    (* maximum palette index found in IDAT *)
{.$endif}
    
    num_trans: png_uint_16;     (* number of transparency values *)
    compression: png_byte;      (* file compression type (always 0) *)
    filter: png_byte;           (* file filter type (always 0) *)
    interlaced: png_byte;       (* PNG_INTERLACE_NONE, PNG_INTERLACE_ADAM7 *)
    pass: png_byte;             (* current interlace pass (0 - 6) *)
    do_filter: png_byte;        (* row filter flags (see PNG_FILTER_ below ) *)
    color_type: png_byte;       (* color type of file *)
    bit_depth: png_byte;        (* bit depth of file *)
    usr_bit_depth: png_byte;    (* bit depth of users row *)
    pixel_depth: png_byte;      (* number of bits per pixel *)
    channels: png_byte;         (* number of channels in file *)
    usr_channels: png_byte;     (* channels at start of write *)
    sig_bytes: png_byte;        (* magic bytes read/written from start of file *)
    maximum_pixel_depth: png_byte; 
                                (* pixel depth used for the row buffers *)
    transformed_pixel_depth: png_byte;
                                (* pixel depth after read/write transforms *)
    io_chunk_string: array [0..4] of png_byte;
                                (* string name of chunk *)

{.$if defined(PNG_READ_FILLER_SUPPORTED) or defined(PNG_WRITE_FILLER_SUPPORTED)}
   filler: png_uint_16;         (* filler bytes for pixel expansion *)
{.$endif}

{.$if defined(PNG_bKGD_SUPPORTED) or defined(PNG_READ_BACKGROUND_SUPPORTED) or
   defined(PNG_READ_ALPHA_MODE_SUPPORTED)}
    background_gamma_type: png_byte;
    background_gamma: png_fixed_point;
    background: png_color_16;   (* background color in screen gamma space *)
{.$ifdef PNG_READ_GAMMA_SUPPORTED}
    background_1: png_color_16; (* background normalized to gamma 1.0 *)
{.$endif}
{.$endif} (* PNG_bKGD_SUPPORTED *)

{.$ifdef PNG_WRITE_FLUSH_SUPPORTED}
    output_flush_fn: png_flush_ptr;  (* Function for flushing output *)
    flush_dist: png_uint_32;    (* how many rows apart to flush, 0 - no flush *)
    flush_rows: png_uint_32;    (* number of rows written since last flush *)
{.$endif}

{.$ifdef PNG_READ_GAMMA_SUPPORTED}
    gamma_shift: integer;       (* number of "insignificant" bits 16-bit gamma *)
    gamma: png_fixed_point;     (* file gamma value *)
    screen_gamma: png_fixed_point; (* screen gamma value (display_exponent) *)

    gamma_table: png_bytep;     (* gamma table for 8-bit depth files *)
    gamma_16_table: png_uint_16pp; (* gamma table for 16-bit depth files *)
{.$if defined(PNG_READ_BACKGROUND_SUPPORTED) or
     defined(PNG_READ_ALPHA_MODE_SUPPORTED) or
     defined(PNG_READ_RGB_TO_GRAY_SUPPORTED)}    
    gamma_from_1: png_bytep;    (* converts from 1.0 to screen *)
    gamma_to_1: png_bytep;      (* converts from file to 1.0 *)
    gamma_16_from_1: png_uint_16pp;  (* converts from 1.0 to screen *)
    gamma_16_to_1: png_uint_16pp;  (* converts from file to 1.0 *)
{.$endif} (* READ_BACKGROUND || READ_ALPHA_MODE || RGB_TO_GRAY *)
{.$endif}

{.$if defined(PNG_READ_GAMMA_SUPPORTED) or defined(PNG_sBIT_SUPPORTED)}
    sig_bit: png_color_8;       (* significant bits in each available channel *)
{.$endif}

{.$if defined(PNG_READ_SHIFT_SUPPORTED) or defined(PNG_WRITE_SHIFT_SUPPORTED)}
    shift: png_color_8;         (* shift for significant bit tranformation *)
{.$endif}

{.$if defined(PNG_tRNS_SUPPORTED) or defined(PNG_READ_BACKGROUND_SUPPORTED) or
      defined(PNG_READ_EXPAND_SUPPORTED) or defined(PNG_READ_BACKGROUND_SUPPORTED)}
    trans_alpha: png_bytep;           (* transparency values for paletted files *)
    trans_color: png_color_16; (* transparency values for non-paletted files *)
{.$endif}

    read_row_fn: png_read_status_ptr;   (* called after each row is decoded *)
    write_row_fn: png_write_status_ptr; (* called after each row is encoded *)
{.$ifdef PNG_PROGRESSIVE_READ_SUPPORTED}
    info_fn: png_progressive_info_ptr; (* called after header data fully read *)
    row_fn: png_progressive_row_ptr;   (* called after each prog. row is decoded *)
    end_fn: png_progressive_end_ptr;   (* called after image is complete *)
    save_buffer_ptr: png_bytep;        (* current location in save_buffer *)
    save_buffer: png_bytep;            (* buffer for previously read data *)
    current_buffer_ptr: png_bytep;     (* current location in current_buffer *)
    current_buffer: png_bytep;         (* buffer for recently used data *)
    push_length: png_uint_32;          (* size of current input chunk *)
    skip_length: png_uint_32;          (* bytes to skip in input data *)
    save_buffer_size: png_size_t;      (* amount of data now in save_buffer *)
    save_buffer_max: png_size_t;       (* total size of save_buffer *)
    buffer_size: png_size_t;           (* total amount of available input data *)
    current_buffer_size: png_size_t;   (* amount of data now in current_buffer *)
    process_mode: integer;             (* what push library is currently doing *)
    cur_palette: integer;              (* current push library palette index *)

{.$endif} (* PNG_PROGRESSIVE_READ_SUPPORTED *)

{.$if defined(__TURBOC__) && !defined(_Windows) && !defined(__FLAT__)}
(* For the Borland special 64K segment handler *)
    offset_table_ptr: png_bytepp;
    offset_table: png_bytep;
    offset_table_number: png_uint_16;
    offset_table_count: png_uint_16;
    offset_table_count_free: png_uint_16;
{.$endif}

{.$ifdef PNG_READ_QUANTIZE_SUPPORTED}
    palette_lookup: png_bytep;         (* lookup table for dithering *)
    quantize_index: png_bytep;         (* index translation for palette files *)
{.$endif}

{.$if defined(PNG_READ_QUANTIZE_SUPPORTED) or defined(PNG_hIST_SUPPORTED)}
    hist: png_uint_16p;                (* histogram *)
{.$endif}

{.$ifdef PNG_WRITE_WEIGHTED_FILTER_SUPPORTED}
    heuristic_method: png_byte;        (* heuristic for row filter selection *)
    num_prev_filters: png_byte;        (* number of weights for previous rows *)
    prev_filters: png_bytep;           (* filter type(s) of previous row(s) *)
    filter_weights: png_uint_16p;      (* weight(s) for previous line(s) *)
    inv_filter_weights: png_uint_16p;  (* 1/weight(s) for previous line(s) *)
    filter_costs: png_uint_16p;        (* relative filter calculation cost *)
    inv_filter_costs: png_uint_16p;    (* 1/relative filter calculation cost *)
{.$endif}

{.$ifdef PNG_TIME_RFC1123_SUPPORTED}
    time_buffer: array [0..28] of AnsiChar; (* String to hold RFC 1123 time text *)
{.$endif}

    (* New members added in libpng-1.0.6 *)

    free_me: png_uint_32;       (* flags items libpng is responsible for freeing *)

{.$ifdef PNG_USER_CHUNKS_SUPPORTED}
    user_chunk_ptr: png_voidp;
    read_user_chunk_fn: png_user_chunk_ptr;  (* user read chunk handler *)
{.$endif}

{.$ifdef PNG_HANDLE_AS_UNKNOWN_SUPPORTED}
    num_chunk_list: integer;
    chunk_list: png_bytep;
{.$endif}

{.$ifdef PNG_READ_sRGB_SUPPORTED}
    (* Added in 1.5.5 to record an sRGB chunk in the png. *)
    is_sRGB: png_byte;
{.$endif}

    (* New members added in libpng-1.0.3 *)
{.$ifdef PNG_READ_RGB_TO_GRAY_SUPPORTED}
    rgb_to_gray_status: png_byte;
    (* Added in libpng 1.5.5 to record setting of coefficients: *)
    rgb_to_gray_coefficients_set: png_byte;
    (* These were changed from png_byte in libpng-1.0.6 *)
    rgb_to_gray_red_coeff: png_uint_16;
    rgb_to_gray_green_coeff: png_uint_16;
    (* deleted in 1.5.5: rgb_to_gray_blue_coeff; *)
{.$endif}
    
    (* New member added in libpng-1.0.4 (renamed in 1.0.9) *)
{.$if defined(PNG_MNG_FEATURES_SUPPORTED)}
    (* Changed from png_byte to png_uint_32 at version 1.2.0 *)
    mng_features_permitted: png_uint_32;
{.$endif}

    (* New member added in libpng-1.0.9, ifdef'ed out in 1.0.12, enabled in 1.2.0 *)
{.$ifdef PNG_MNG_FEATURES_SUPPORTED}
    filter_type: png_byte;
{.$endif}

    (* New members added in libpng-1.2.0 *)

    (* New members added in libpng-1.0.2 but first enabled by default in 1.2.0 *)
{.$ifdef PNG_USER_MEM_SUPPORTED}
    mem_ptr: png_voidp;                (* user supplied struct for mem functions *)
    malloc_fn: png_malloc_ptr;         (* function for allocating memory *)
    free_fn: png_free_ptr;             (* function for freeing memory *)
{.$endif}

    (* New member added in libpng-1.0.13 and 1.2.0 *)
    big_row_buf: png_bytep;            (* buffer to save current (unfiltered) row *)

{.$ifdef PNG_READ_QUANTIZE_SUPPORTED}
    (* The following three members were added at version 1.0.14 and 1.2.4 *)
    quantize_sort: png_bytep;          (* working sort array *)
    index_to_palette: png_bytep;       (* where the original index currently is *)
                                       (* in the palette *)
    palette_to_index: png_bytep;       (* which original index points to this *)
                                       (* palette color *)
{.$endif}

    (* New members added in libpng-1.0.16 and 1.2.6 *)
    compression_type: png_byte;

{.$ifdef PNG_USER_LIMITS_SUPPORTED}
    user_width_max: png_uint_32;
    user_height_max: png_uint_32;
    
    (* Added in libpng-1.4.0: Total number of sPLT, text, and unknown
    * chunks that can be stored (0 means unlimited).
    *)
    user_chunk_cache_max: png_uint_32;

    (* Total memory that a zTXt, sPLT, iTXt, iCCP, or unknown chunk
     * can occupy when decompressed.  0 means unlimited.
     *)
    user_chunk_malloc_max: png_alloc_size_t;
{.$endif}

    (* New member added in libpng-1.0.25 and 1.2.17 *)
{.$ifdef PNG_UNKNOWN_CHUNKS_SUPPORTED}
    (* Storage for unknown chunk that the library doesn't recognize. *)
    unknown_chunk: png_unknown_chunk;
{.$endif}

    (* New member added in libpng-1.2.26 *)
    old_big_row_buf_size: png_size_t;

    (* New member added in libpng-1.2.30 *)
    chunkdata: png_charp;  (* buffer for reading chunk data *)

{.$ifdef PNG_IO_STATE_SUPPORTED}
    (* New member added in libpng-1.4.0 *)
    io_state: png_uint_32;
{.$endif}

    (* New member added in libpng-1.5.6 *)
    big_prev_row: png_bytep;

   //void (*read_filter[PNG_FILTER_VALUE_LAST-1])(png_row_infop row_info,
   //   png_bytep row, png_const_bytep prev_row);
   read_filter: pointer;
  end;
  png_structpp = ^png_structp;

(* Section 3: exported functions
 * Here are the function definitions most commonly used.  This is not
 * the place to find out how to use libpng.  See libpng-manual.txt for the
 * full explanation, see example.c for the summary.  This just provides
 * a simple one line description of the use of each function.
 *
 * The PNG_EXPORT() and PNG_EXPORTA() macros used below are defined in
 * pngconf.h and in the *.dfn files in the scripts directory.
 *
 *   PNG_EXPORT(ordinal, type, name, (args));
 *
 *       ordinal:    ordinal that is used while building
 *                   *.def files. The ordinal value is only
 *                   relevant when preprocessing png.h with
 *                   the *.dfn files for building symbol table
 *                   entries, and are removed by pngconf.h.
 *       type:       return type of the function
 *       name:       function name
 *       args:       function arguments, with types
 *
 * When we wish to append attributes to a function prototype we use
 * the PNG_EXPORTA() macro instead.
 *
 *   PNG_EXPORTA(ordinal, type, name, (args), attributes);
 *
 *       ordinal, type, name, and args: same as in PNG_EXPORT().
 *       attributes: function attributes
 *)

{.$IFNDEF LIB_PNG_STATIC_LINK}

var
(* Returns the version number of the library *)
  png_access_version_number: function: png_uint_32; cdecl;

(* Tell lib we have already handled the first <num_bytes> magic bytes.
  Handling more than 8 bytes from the beginning of the file is an error.*)
  png_set_sig_bytes: procedure(png_ptr: png_structp; num_bytes: integer); cdecl;
  
(* Check sig[start] through sig[start + num_to_check - 1] to see if it's a
  PNG file.  Returns zero if the supplied bytes match the 8-byte PNG
  signature, and non-zero otherwise.  Having num_to_check == 0 or
  start > 7 will always fail (ie return non-zero). *)
  png_sig_cmp: function(sig: png_const_bytep; start: png_size_t; num_to_check: png_size_t): integer; cdecl;
  
(* Simple signature checking function.  This is the same as calling
  png_check_sig(sig, n) := !png_sig_cmp(sig, 0, n). *)
  png_check_sig: function(sig: png_bytep; num: integer): integer; cdecl;
  
(* Allocate and initialize png_ptr struct for reading, and any other memory. *)
  png_create_read_struct: function(user_png_ver: png_const_charp; error_ptr: png_voidp; error_fn: png_error_ptr; warn_fn: png_error_ptr): png_structp; cdecl;

(* Allocate and initialize png_ptr struct for writing, and any other memory *)
  png_create_write_struct: function(user_png_ver: png_const_charp; error_ptr: png_voidp; error_fn: png_error_ptr; warn_fn: png_error_ptr): png_structp; cdecl;

  png_get_compression_buffer_size: function(png_ptr: png_structp): png_size_t; cdecl;
  
  png_set_compression_buffer_size: procedure(png_ptr: png_structp; size: png_size_t); cdecl;
  
(* Moved from pngconf.h in 1.4.0 and modified to ensure setjmp/longjmp
 * match up.
 *)
{.$ifdef PNG_SETJMP_SUPPORTED}
(* This function returns the jmp_buf built in to *png_ptr.  It must be
 * supplied with an appropriate 'longjmp' function to use on that jmp_buf
 * unless the default error function is overridden in which case NULL is
 * acceptable.  The size of the jmp_buf is checked against the actual size
 * allocated by the library - the call will return NULL on a mismatch
 * indicating an ABI mismatch.
 *)
//PNG_EXPORT(8, jmp_buf*, png_set_longjmp_fn, (png_structp png_ptr,
//    png_longjmp_ptr longjmp_fn, size_t jmp_buf_size));
//#  define png_jmpbuf(png_ptr) \
//      (*png_set_longjmp_fn((png_ptr), longjmp, sizeof (jmp_buf)))
{.$else}
//#  define png_jmpbuf(png_ptr) \
//      (LIBPNG_WAS_COMPILED_WITH__PNG_NO_SETJMP)
//#endif
(* This function should be used by libpng applications in place of
 * longjmp(png_ptr->jmpbuf, val).  If longjmp_fn() has been set, it
 * will use it; otherwise it will call PNG_ABORT().  This function was
 * added in libpng-1.5.0.
 *)
  png_longjmp: procedure(png_ptr: png_structp; val: integer); cdecl;

{.$ifdef PNG_READ_SUPPORTED}
(* Reset the compression stream *)
  png_reset_zstream: function(png_ptr: png_structp): integer; cdecl;
{.$endif}

(* New functions added in libpng-1.0.2 (not enabled by default until 1.2.0) *)
{.$ifdef PNG_USER_MEM_SUPPORTED}
  png_create_read_struct_2: function(user_png_ver: png_const_charp; error_ptr: png_voidp; error_fn: png_error_ptr; warn_fn: png_error_ptr; mem_ptr: png_voidp; malloc_fn: png_malloc_ptr; free_fn: png_free_ptr): png_structp; cdecl;
  png_create_write_struct_2: function(user_png_ver: png_const_charp; error_ptr: png_voidp; error_fn: png_error_ptr; warn_fn: png_error_ptr; mem_ptr: png_voidp; malloc_fn: png_malloc_ptr; free_fn: png_free_ptr): png_structp; cdecl;
{.$endif}

(* Write the PNG file signature. *)
  png_write_sig: procedure(png_ptr: png_structp); cdecl;
  
(* Write a PNG chunk - size, type, (optional) data, CRC. *)
  png_write_chunk: procedure (png_ptr: png_structp; chunk_name: png_const_bytep; data: png_const_bytep; length: png_size_t); cdecl;

(* Write the start of a PNG chunk - length and chunk name. *)
  png_write_chunk_start: procedure(png_ptr: png_structp; chunk_name: png_const_bytep; length: png_uint_32); cdecl;

(* Write the data of a PNG chunk started with png_write_chunk_start(). *)
  png_write_chunk_data: procedure(png_ptr: png_structp; data: png_const_bytep; length: png_size_t); cdecl;

(* Finish a chunk started with png_write_chunk_start() (includes CRC). *)
  png_write_chunk_end: procedure(png_ptr: png_structp); cdecl;

(* Allocate and initialize the info structure *)
  png_create_info_struct: function(png_ptr: png_structp): png_infop; cdecl;

(* Initialize the info structure (old interface - DEPRECATED) *)
  png_info_init: procedure(info_ptr: png_infop); cdecl;

  png_info_init_3: procedure(info_ptr: png_infopp; png_info_struct_size: png_size_t); cdecl;

(* Writes all the PNG information before the image. *)
  png_write_info_before_PLTE: procedure(png_ptr: png_structp; info_ptr: png_infop); cdecl;
  png_write_info: procedure(png_ptr: png_structp; info_ptr: png_infop); cdecl;

{.$ifdef PNG_SEQUENTIAL_READ_SUPPORTED}
(* Read the information before the actual image data. *)
  png_read_info: procedure(png_ptr: png_structp; info_ptr: png_infop); cdecl;
{.$endif}

//#ifdef PNG_TIME_RFC1123_SUPPORTED
  png_convert_to_rfc1123: function(png_ptr: png_structp; ptime: png_const_timep): png_const_charp; cdecl;
{.$endif}
  
//#ifdef PNG_CONVERT_tIME_SUPPORTED
(* Convert from a struct tm to png_time *)
  png_convert_from_struct_tm: procedure(ptime: png_timep; ttime: TMp); cdecl;

(* convert from time_t to png_time.  Uses gmtime() *)
  png_convert_from_time_t: procedure(ptime: png_timep; ttime: time_t);
//#endif /* PNG_CONVERT_tIME_SUPPORTED */

//#ifdef PNG_READ_EXPAND_SUPPORTED
(* Expand data to 24-bit RGB, or 8-bit grayscale, with alpha if available. *)
  png_set_expand: procedure(png_ptr: png_structp); cdecl;
  png_set_expand_gray_1_2_4_to_8: procedure(png_ptr: png_structp); cdecl;
  png_set_palette_to_rgb: procedure(png_ptr: png_structp); cdecl;
  png_set_tRNS_to_alpha: procedure(png_ptr: png_structp); cdecl;
//#endif

//#ifdef PNG_READ_EXPAND_16_SUPPORTED
(* Expand to 16-bit channels, forces conversion of palette to RGB and expansion
 * of a tRNS chunk if present.
 *)
  png_set_expand_16: procedure(png_ptr: png_structp); cdecl;
//#endif

//#if defined(PNG_READ_BGR_SUPPORTED) || defined(PNG_WRITE_BGR_SUPPORTED)  
(* Use blue, green, red order for pixels. *)
  png_set_bgr: procedure(png_ptr: png_structp); cdecl;
//#endif

//#ifdef PNG_READ_GRAY_TO_RGB_SUPPORTED
(* Expand the grayscale to 24-bit RGB if necessary. *)
  png_set_gray_to_rgb: procedure(png_ptr: png_structp); cdecl;
//#endif

//#ifdef PNG_READ_RGB_TO_GRAY_SUPPORTED
(* Reduce RGB to grayscale. *)
const
  PNG_ERROR_ACTION_NONE   = 1;
  PNG_ERROR_ACTION_WARN   = 2;
  PNG_ERROR_ACTION_ERROR  = 3;
  PNG_RGB_TO_GRAY_DEFAULT = (-1); (*for red/green coefficients*)
var
  png_set_rgb_to_gray: procedure(png_ptr: png_structp; error_action: integer; red: double; green: double); cdecl;
  png_set_rgb_to_gray_fixed: procedure(png_ptr: png_structp; error_action: integer; red: png_fixed_point; green: png_fixed_point); cdecl;
  png_get_rgb_to_gray_status: function(png_ptr: png_structp): png_byte; cdecl;
//#endif

//#ifdef PNG_BUILD_GRAYSCALE_PALETTE_SUPPORTED
  png_build_grayscale_palette: procedure(bit_depth: integer; palette: png_colorp); cdecl;
//#endif

//#ifdef PNG_READ_ALPHA_MODE_SUPPORTED
(* How the alpha channel is interpreted - this affects how the color channels of
 * a PNG file are returned when an alpha channel, or tRNS chunk in a palette
 * file, is present.
 *
 * This has no effect on the way pixels are written into a PNG output
 * datastream. The color samples in a PNG datastream are never premultiplied
 * with the alpha samples.
 *
 * The default is to return data according to the PNG specification: the alpha
 * channel is a linear measure of the contribution of the pixel to the
 * corresponding composited pixel.  The gamma encoded color channels must be
 * scaled according to the contribution and to do this it is necessary to undo
 * the encoding, scale the color values, perform the composition and reencode
 * the values.  This is the 'PNG' mode.
 *
 * The alternative is to 'associate' the alpha with the color information by
 * storing color channel values that have been scaled by the alpha.  The
 * advantage is that the color channels can be resampled (the image can be
 * scaled) in this form.  The disadvantage is that normal practice is to store
 * linear, not (gamma) encoded, values and this requires 16-bit channels for
 * still images rather than the 8-bit channels that are just about sufficient if
 * gamma encoding is used.  In addition all non-transparent pixel values,
 * including completely opaque ones, must be gamma encoded to produce the final
 * image.  This is the 'STANDARD', 'ASSOCIATED' or 'PREMULTIPLIED' mode (the
 * latter being the two common names for associated alpha color channels.)
 *
 * Since it is not necessary to perform arithmetic on opaque color values so
 * long as they are not to be resampled and are in the final color space it is
 * possible to optimize the handling of alpha by storing the opaque pixels in
 * the PNG format (adjusted for the output color space) while storing partially
 * opaque pixels in the standard, linear, format.  The accuracy required for
 * standard alpha composition is relatively low, because the pixels are
 * isolated, therefore typically the accuracy loss in storing 8-bit linear
 * values is acceptable.  (This is not true if the alpha channel is used to
 * simulate transparency over large areas - use 16 bits or the PNG mode in
 * this case!)  This is the 'OPTIMIZED' mode.  For this mode a pixel is
 * treated as opaque only if the alpha value is equal to the maximum value.
 *
 * The final choice is to gamma encode the alpha channel as well.  This is
 * broken because, in practice, no implementation that uses this choice
 * correctly undoes the encoding before handling alpha composition.  Use this
 * choice only if other serious errors in the software or hardware you use
 * mandate it; the typical serious error is for dark halos to appear around
 * opaque areas of the composited PNG image because of arithmetic overflow.
 *
 * The API function png_set_alpha_mode specifies which of these choices to use
 * with an enumerated 'mode' value and the gamma of the required output:
 *)
const
  PNG_ALPHA_PNG           = 0; (* according to the PNG standard *)
  PNG_ALPHA_STANDARD      = 1; (* according to Porter/Duff *)
  PNG_ALPHA_ASSOCIATED    = 1; (* as above; this is the normal practice *)
  PNG_ALPHA_PREMULTIPLIED = 1; (* as above *)
  PNG_ALPHA_OPTIMIZED     = 2; (* 'PNG' for opaque pixels, else 'STANDARD' *)
  PNG_ALPHA_BROKEN        = 3; (* the alpha channel is gamma encoded *)
var
  png_set_alpha_mode: procedure(png_ptr: png_structp; mode: integer; output_gamma: double); cdecl;
  png_set_alpha_mode_fixed: procedure(png_ptr: png_structp; mode: integer; output_gamma: png_fixed_point); cdecl;
//#endif

//#if defined(PNG_READ_GAMMA_SUPPORTED) || defined(PNG_READ_ALPHA_MODE_SUPPORTED)
(* The output_gamma value is a screen gamma in libpng terminology: it expresses
 * how to decode the output values, not how they are encoded.  The values used
 * correspond to the normal numbers used to describe the overall gamma of a
 * computer display system; for example 2.2 for an sRGB conformant system.  The
 * values are scaled by 100000 in the _fixed version of the API (so 220000 for
 * sRGB.)
 *
 * The inverse of the value is always used to provide a default for the PNG file
 * encoding if it has no gAMA chunk and if png_set_gamma() has not been called
 * to override the PNG gamma information.
 *
 * When the ALPHA_OPTIMIZED mode is selected the output gamma is used to encode
 * opaque pixels however pixels with lower alpha values are not encoded,
 * regardless of the output gamma setting.
 *
 * When the standard Porter Duff handling is requested with mode 1 the output
 * encoding is set to be linear and the output_gamma value is only relevant
 * as a default for input data that has no gamma information.  The linear output
 * encoding will be overridden if png_set_gamma() is called - the results may be
 * highly unexpected!
 *
 * The following numbers are derived from the sRGB standard and the research
 * behind it.  sRGB is defined to be approximated by a PNG gAMA chunk value of
 * 0.45455 (1/2.2) for PNG.  The value implicitly includes any viewing
 * correction required to take account of any differences in the color
 * environment of the original scene and the intended display environment; the
 * value expresses how to *decode* the image for display, not how the original
 * data was *encoded*.
 *
 * sRGB provides a peg for the PNG standard by defining a viewing environment.
 * sRGB itself, and earlier TV standards, actually use a more complex transform
 * (a linear portion then a gamma 2.4 power law) than PNG can express.  (PNG is
 * limited to simple power laws.)  By saying that an image for direct display on
 * an sRGB conformant system should be stored with a gAMA chunk value of 45455
 * (11.3.3.2 and 11.3.3.5 of the ISO PNG specification) the PNG specification
 * makes it possible to derive values for other display systems and
 * environments.
 *
 * The Mac value is deduced from the sRGB based on an assumption that the actual
 * extra viewing correction used in early Mac display systems was implemented as
 * a power 1.45 lookup table.
 *
 * Any system where a programmable lookup table is used or where the behavior of
 * the final display device characteristics can be changed requires system
 * specific code to obtain the current characteristic.  However this can be
 * difficult and most PNG gamma correction only requires an approximate value.
 *
 * By default, if png_set_alpha_mode() is not called, libpng assumes that all
 * values are unencoded, linear, values and that the output device also has a
 * linear characteristic.  This is only very rarely correct - it is invariably
 * better to call png_set_alpha_mode() with PNG_DEFAULT_sRGB than rely on the
 * default if you don't know what the right answer is!
 *
 * The special value PNG_GAMMA_MAC_18 indicates an older Mac system (pre Mac OS
 * 10.6) which used a correction table to implement a somewhat lower gamma on an
 * otherwise sRGB system.
 *
 * Both these values are reserved (not simple gamma values) in order to allow
 * more precise correction internally in the future.
 *
 * NOTE: the following values can be passed to either the fixed or floating
 * point APIs, but the floating point API will also accept floating point
 * values.
 *)
const
  PNG_DEFAULT_sRGB        = -1;       (* sRGB gamma and color space *)
  PNG_GAMMA_MAC_18        = -2;       (* Old Mac '1.8' gamma and color space *)
  PNG_GAMMA_sRGB          = 220000;   (* Television standards--matches sRGB gamma *)
  PNG_GAMMA_LINEAR        = PNG_FP_1; (* Linear *)
//#endif

(* The following are examples of calls to png_set_alpha_mode to achieve the
 * required overall gamma correction and, where necessary, alpha
 * premultiplication.
 *
 * png_set_alpha_mode(pp, PNG_ALPHA_PNG, PNG_DEFAULT_sRGB);
 *    This is the default libpng handling of the alpha channel - it is not
 *    pre-multiplied into the color components.  In addition the call states
 *    that the output is for a sRGB system and causes all PNG files without gAMA
 *    chunks to be assumed to be encoded using sRGB.
 *
 * png_set_alpha_mode(pp, PNG_ALPHA_PNG, PNG_GAMMA_MAC);
 *    In this case the output is assumed to be something like an sRGB conformant
 *    display preceeded by a power-law lookup table of power 1.45.  This is how
 *    early Mac systems behaved.
 *
 * png_set_alpha_mode(pp, PNG_ALPHA_STANDARD, PNG_GAMMA_LINEAR);
 *    This is the classic Jim Blinn approach and will work in academic
 *    environments where everything is done by the book.  It has the shortcoming
 *    of assuming that input PNG data with no gamma information is linear - this
 *    is unlikely to be correct unless the PNG files where generated locally.
 *    Most of the time the output precision will be so low as to show
 *    significant banding in dark areas of the image.
 *
 * png_set_expand_16(pp);
 * png_set_alpha_mode(pp, PNG_ALPHA_STANDARD, PNG_DEFAULT_sRGB);
 *    This is a somewhat more realistic Jim Blinn inspired approach.  PNG files
 *    are assumed to have the sRGB encoding if not marked with a gamma value and
 *    the output is always 16 bits per component.  This permits accurate scaling
 *    and processing of the data.  If you know that your input PNG files were
 *    generated locally you might need to replace PNG_DEFAULT_sRGB with the
 *    correct value for your system.
 *
 * png_set_alpha_mode(pp, PNG_ALPHA_OPTIMIZED, PNG_DEFAULT_sRGB);
 *    If you just need to composite the PNG image onto an existing background
 *    and if you control the code that does this you can use the optimization
 *    setting.  In this case you just copy completely opaque pixels to the
 *    output.  For pixels that are not completely transparent (you just skip
 *    those) you do the composition math using png_composite or png_composite_16
 *    below then encode the resultant 8-bit or 16-bit values to match the output
 *    encoding.
 *
 * Other cases
 *    If neither the PNG nor the standard linear encoding work for you because
 *    of the software or hardware you use then you have a big problem.  The PNG
 *    case will probably result in halos around the image.  The linear encoding
 *    will probably result in a washed out, too bright, image (it's actually too
 *    contrasty.)  Try the ALPHA_OPTIMIZED mode above - this will probably
 *    substantially reduce the halos.  Alternatively try:
 *
 * png_set_alpha_mode(pp, PNG_ALPHA_BROKEN, PNG_DEFAULT_sRGB);
 *    This option will also reduce the halos, but there will be slight dark
 *    halos round the opaque parts of the image where the background is light.
 *    In the OPTIMIZED mode the halos will be light halos where the background
 *    is dark.  Take your pick - the halos are unavoidable unless you can get
 *    your hardware/software fixed!  (The OPTIMIZED approach is slightly
 *    faster.)
 *
 * When the default gamma of PNG files doesn't match the output gamma.
 *    If you have PNG files with no gamma information png_set_alpha_mode allows
 *    you to provide a default gamma, but it also sets the ouput gamma to the
 *    matching value.  If you know your PNG files have a gamma that doesn't
 *    match the output you can take advantage of the fact that
 *    png_set_alpha_mode always sets the output gamma but only sets the PNG
 *    default if it is not already set:
 *
 * png_set_alpha_mode(pp, PNG_ALPHA_PNG, PNG_DEFAULT_sRGB);
 * png_set_alpha_mode(pp, PNG_ALPHA_PNG, PNG_GAMMA_MAC);
 *    The first call sets both the default and the output gamma values, the
 *    second call overrides the output gamma without changing the default.  This
 *    is easier than achieving the same effect with png_set_gamma.  You must use
 *    PNG_ALPHA_PNG for the first call - internal checking in png_set_alpha will
 *    fire if more than one call to png_set_alpha_mode and png_set_background is
 *    made in the same read operation, however multiple calls with PNG_ALPHA_PNG
 *    are ignored.
 *)
var 
//#ifdef PNG_READ_STRIP_ALPHA_SUPPORTED 
  png_set_strip_alpha: procedure(png_ptr: png_structp); cdecl;
//#endif

//#if defined(PNG_READ_SWAP_ALPHA_SUPPORTED) || \
    //defined(PNG_WRITE_SWAP_ALPHA_SUPPORTED)
  png_set_swap_alpha: procedure(png_ptr: png_structp); cdecl;
//#endif

//#if defined(PNG_READ_INVERT_ALPHA_SUPPORTED) || \
    //defined(PNG_WRITE_INVERT_ALPHA_SUPPORTED)
  png_set_invert_alpha: procedure(png_ptr: png_structp); cdecl;
//#endif

//#if defined(PNG_READ_FILLER_SUPPORTED) || defined(PNG_WRITE_FILLER_SUPPORTED)
(* Add a filler byte to 8-bit Gray or 24-bit RGB images. *)
  png_set_filler: procedure(png_ptr: png_structp; filler: png_uint_32; flags: integer); cdecl;

const
  (* The values of the PNG_FILLER_ defines should NOT be changed *)
  PNG_FILLER_BEFORE = 0;
  PNG_FILLER_AFTER  = 1;

var
(* Add an alpha byte to 8-bit Gray or 24-bit RGB images. *)
  png_set_add_alpha: procedure(png_ptr: png_structp; filler: png_uint_32; flags: integer); cdecl;
//#endif /* PNG_READ_FILLER_SUPPORTED || PNG_WRITE_FILLER_SUPPORTED */

//#if defined(PNG_READ_SWAP_SUPPORTED) || defined(PNG_WRITE_SWAP_SUPPORTED)
(* Swap bytes in 16-bit depth files. *)
  png_set_swap: procedure(png_ptr: png_structp); cdecl;
//#endif

//#if defined(PNG_READ_PACK_SUPPORTED) || defined(PNG_WRITE_PACK_SUPPORTED)
(* Use 1 byte per pixel in 1, 2, or 4-bit depth files. *)
  png_set_packing: procedure(png_ptr: png_structp); cdecl;
//#endif

//#if defined(PNG_READ_PACKSWAP_SUPPORTED) || \
    //defined(PNG_WRITE_PACKSWAP_SUPPORTED)
(* Swap packing order of pixels in bytes. *)
  png_set_packswap: procedure(png_ptr: png_structp); cdecl;
//#endif

//#if defined(PNG_READ_SHIFT_SUPPORTED) || defined(PNG_WRITE_SHIFT_SUPPORTED)
(* Converts files to legal bit depths. *)
  png_set_shift: procedure(png_ptr: png_structp; true_bits: png_color_8p); cdecl;
//#endif

//#if defined(PNG_READ_INTERLACING_SUPPORTED) || \
    //defined(PNG_WRITE_INTERLACING_SUPPORTED)
(* Have the code handle the interlacing.  Returns the number of passes.
 * MUST be called before png_read_update_info or png_start_read_image,
 * otherwise it will not have the desired effect.  Note that it is still
 * necessary to call png_read_row or png_read_rows png_get_image_height
 * times for each pass.
*)
  png_set_interlace_handling: function(png_ptr: png_structp): integer;
//#endif

//#if defined(PNG_READ_INVERT_SUPPORTED) || defined(PNG_WRITE_INVERT_SUPPORTED)
(* Invert monochrome files *)
  png_set_invert_mono: procedure(png_ptr: png_structp); cdecl;
//#endif

//#ifdef PNG_READ_BACKGROUND_SUPPORTED
(* Handle alpha and tRNS by replacing with a background color.  Prior to
 * libpng-1.5.4 this API must not be called before the PNG file header has been
 * read.  Doing so will result in unexpected behavior and possible warnings or
 * errors if the PNG file contains a bKGD chunk.
 *)
  png_set_background: procedure(png_ptr: png_structp; background_color: png_const_color_16p; background_gamma_code: integer; need_expand: integer; background_gamma: double); cdecl;
  png_set_background_fixed: procedure(png_ptr: png_structp; background_color: png_const_color_16p; background_gamma_code: integer; need_expand: integer; background_gamma: png_fixed_point); cdecl;

const
  PNG_BACKGROUND_GAMMA_UNKNOWN = 0;
  PNG_BACKGROUND_GAMMA_SCREEN  = 1;
  PNG_BACKGROUND_GAMMA_FILE    = 2;
  PNG_BACKGROUND_GAMMA_UNIQUE  = 3;
//#endif

var
//#ifdef PNG_READ_SCALE_16_TO_8_SUPPORTED
(* Scale a 16-bit depth file down to 8-bit, accurately. *)
  png_set_scale_16: procedure(png_ptr: png_structp);
//#endif

//#ifdef PNG_READ_STRIP_16_TO_8_SUPPORTED
//#define PNG_READ_16_TO_8 SUPPORTED /* Name prior to 1.5.4 */
(* Strip the second byte of information from a 16-bit depth file. *)
  png_set_strip_16: procedure(png_ptr: png_structp); cdecl;
//#endif

//#ifdef PNG_READ_QUANTIZE_SUPPORTED
(* Turn on quantizing, and reduce the palette to the number of colors
 * available.
 *)
  png_set_quantize: procedure(png_ptr: png_structp; palette: png_colorp; num_palette: integer; maximum_colors: integer; histogram: png_const_uint_16p; full_quantize: integer); cdecl;
//#endif

//#ifdef PNG_READ_GAMMA_SUPPORTED
(* The threshold on gamma processing is configurable but hard-wired into the
 * library.  The following is the floating point variant.
 *)
const
 PNG_GAMMA_THRESHOLD = (PNG_GAMMA_THRESHOLD_FIXED * 0.00001);

(* Handle gamma correction. Screen_gamma=(display_exponent).
 * NOTE: this API simply sets the screen and file gamma values. It will
 * therefore override the value for gamma in a PNG file if it is called after
 * the file header has been read - use with care  - call before reading the PNG
 * file for best results!
 *
 * These routines accept the same gamma values as png_set_alpha_mode (described
 * above).  The PNG_GAMMA_ defines and PNG_DEFAULT_sRGB can be passed to either
 * API (floating point or fixed.)  Notice, however, that the 'file_gamma' value
 * is the inverse of a 'screen gamma' value.
 *)
var
  png_set_gamma: procedure(png_ptr: png_structp; screen_gamma: double; override_file_gamma: double); cdecl;
  png_set_gamma_fixed: procedure(png_ptr: png_structp; screen_gamma: png_fixed_point; override_file_gamma: png_fixed_point); cdecl;
//#endif

//#ifdef PNG_WRITE_FLUSH_SUPPORTED
(* Set how many lines between output flushes - 0 for no flushing *)
  png_set_flush: procedure(png_ptr: png_structp; nrows: integer); cdecl;  
(* Flush the current PNG output buffer *)
  png_write_flush: procedure(png_ptr: png_structp); cdecl;
//#endif

(* Optional update palette with requested transformations *)
  png_start_read_image: procedure(png_ptr: png_structp); cdecl;

(* Optional call to update the users info structure *)
  png_read_update_info: procedure(png_ptr: png_structp; info_ptr: png_infop); cdecl;

//#ifdef PNG_SEQUENTIAL_READ_SUPPORTED
(* read one or more rows of image data. *)
  png_read_rows: procedure(png_ptr: png_structp; row: png_bytepp; display_row: png_bytepp; num_rows: png_uint_32); cdecl;
//#endif

//#ifdef PNG_SEQUENTIAL_READ_SUPPORTED
(* read a row of data. *)
  png_read_row: procedure(png_ptr: png_structp; row: png_bytep; display_row: png_bytep); cdecl;
//#endif

//#ifdef PNG_SEQUENTIAL_READ_SUPPORTED
(* read the whole image into memory at once. *)
  png_read_image: procedure(png_ptr: png_structp; image: png_bytepp); cdecl;
//#endif

(* write a row of image data *)
  png_write_row: procedure(png_ptr: png_structp; row: png_bytep); cdecl;

(* Write a few rows of image data: (*row) is not written; however, the type
 * is declared as writeable to maintain compatibility with previous versions
 * of libpng and to allow the 'display_row' array from read_rows to be passed
 * unchanged to write_rows.
 *)
  png_write_rows: procedure(png_ptr: png_structp; row: png_bytepp; num_rows: png_uint_32); cdecl;

(* write the image data *)
  png_write_image: procedure(png_ptr: png_structp; image: png_bytepp); cdecl;

(* writes the end of the PNG file. *)
  png_write_end: procedure(png_ptr: png_structp; info_ptr: png_infop); cdecl;

//#ifdef PNG_SEQUENTIAL_READ_SUPPORTED
(* read the end of the PNG file. *)
  png_read_end: procedure(png_ptr: png_structp; info_ptr: png_infop); cdecl;
//#endif

(* free any memory associated with the png_info_struct *)
  png_destroy_info_struct: procedure(png_ptr: png_structp; info_ptr_ptr: png_infopp); cdecl;

(* free any memory associated with the png_struct and the png_info_structs *)
  png_destroy_read_struct: procedure(png_ptr_ptr: png_structpp; info_ptr_ptr: png_infopp; end_info_ptr_ptr: png_infopp); cdecl;

(* free any memory associated with the png_struct and the png_info_structs *)
  png_destroy_write_struct: procedure(png_ptr_ptr: png_structpp; info_ptr_ptr: png_infopp); cdecl;

(* set the libpng method of handling chunk CRC errors *)
  png_set_crc_action: procedure (png_ptr: png_structp; crit_action: integer; ancil_action: integer);
  
const
(* Values for png_set_crc_action() to say how to handle CRC errors in
  ancillary and critical chunks, and whether to use the data contained
  therein.  Note that it is impossible to "discard" data in a critical
  chunk.  For versions prior to 0.90, the action was always error/quit,
  whereas in version 0.90 and later, the action for CRC errors in ancillary
  chunks is warn/discard.  These values should NOT be changed.
 
  value                        action:critical     action:ancillary  *)
 
  PNG_CRC_DEFAULT      = 0;  (* error/quit          warn/discard data *)
  PNG_CRC_ERROR_QUIT   = 1;  (* error/quit          error/quit        *)
  PNG_CRC_WARN_DISCARD = 2;  (* (INVALID)           warn/discard data *)
  PNG_CRC_WARN_USE     = 3;  (* warn/use data       warn/use data     *)
  PNG_CRC_QUIET_USE    = 4;  (* quiet/use data      quiet/use data    *)
  PNG_CRC_NO_CHANGE    = 5;  (* use current value   use current value *)

(* These functions give the user control over the scan-line filtering in
  libpng and the compression methods used by zlib.  These functions are
  mainly useful for testing, as the defaults should work with most users.
  Those users who are tight on memory or want faster performance at the
  expense of compression can modify them.  See the compression library
  header file (zlib.h) for an explination of the compression functions. *)

var
(* set the filtering method(s) used by libpng.  Currently, the only valid
  value for "method" is 0.*)
  png_set_filter: procedure(png_ptr: png_structp; method: integer; filters: integer); cdecl;

const
(* Flags for png_set_filter() to say which filters to use.  The flags
  are chosen so that they don't conflict with real filter types
  below, in case they are supplied instead of the #defined constants.
  These values should NOT be changed. *)
  PNG_NO_FILTERS     = $00;
  PNG_FILTER_NONE    = $08;
  PNG_FILTER_SUB     = $10;
  PNG_FILTER_UP      = $20;
  PNG_FILTER_AVG     = $40;
  PNG_FILTER_PAETH   = $80;
  PNG_ALL_FILTERS    = PNG_FILTER_NONE or PNG_FILTER_SUB or PNG_FILTER_UP or PNG_FILTER_AVG or PNG_FILTER_PAETH;

(* Filter values (not flags) - used in pngwrite.c, pngwutil.c for now.
  These defines should NOT be changed. *)
  PNG_FILTER_VALUE_NONE  = 0;
  PNG_FILTER_VALUE_SUB   = 1;
  PNG_FILTER_VALUE_UP    = 2;
  PNG_FILTER_VALUE_AVG   = 3;
  PNG_FILTER_VALUE_PAETH = 4;
  PNG_FILTER_VALUE_LAST  = 5;
  
var
//#ifdef PNG_WRITE_WEIGHTED_FILTER_SUPPORTED (* EXPERIMENTAL *)
(* The "heuristic_method" is given by one of the PNG_FILTER_HEURISTIC_
 * defines, either the default (minimum-sum-of-absolute-differences), or
 * the experimental method (weighted-minimum-sum-of-absolute-differences).
 *
 * Weights are factors >= 1.0, indicating how important it is to keep the
 * filter type consistent between rows.  Larger numbers mean the current
 * filter is that many times as likely to be the same as the "num_weights"
 * previous filters.  This is cumulative for each previous row with a weight.
 * There needs to be "num_weights" values in "filter_weights", or it can be
 * NULL if the weights aren't being specified.  Weights have no influence on
 * the selection of the first row filter.  Well chosen weights can (in theory)
 * improve the compression for a given image.
 *
 * Costs are factors >= 1.0 indicating the relative decoding costs of a
 * filter type.  Higher costs indicate more decoding expense, and are
 * therefore less likely to be selected over a filter with lower computational
 * costs.  There needs to be a value in "filter_costs" for each valid filter
 * type (given by PNG_FILTER_VALUE_LAST), or it can be NULL if you aren't
 * setting the costs.  Costs try to improve the speed of decompression without
 * unduly increasing the compressed image size.
 *
 * A negative weight or cost indicates the default value is to be used, and
 * values in the range [0.0, 1.0) indicate the value is to remain unchanged.
 * The default values for both weights and costs are currently 1.0, but may
 * change if good general weighting/cost heuristics can be found.  If both
 * the weights and costs are set to 1.0, this degenerates the WEIGHTED method
 * to the UNWEIGHTED method, but with added encoding time/computation.
 *)
  png_set_filter_heuristics: procedure(png_ptr: png_structp; heuristic_method: integer; num_weights: integer; filter_weights: png_const_doublep; filter_costs: png_const_doublep); cdecl;
  png_set_filter_heuristics_fixed: procedure(png_ptr: png_structp; heuristic_method: integer; num_weights: integer; filter_weights: png_const_fixed_point_p; filter_costs: png_const_fixed_point_p); cdecl;
//#endif /*  PNG_WRITE_WEIGHTED_FILTER_SUPPORTED */

const
(* Heuristic used for row filter selection.  These defines should NOT be
  changed. *)
  PNG_FILTER_HEURISTIC_DEFAULT    = 0;  (* Currently "UNWEIGHTED" *)
  PNG_FILTER_HEURISTIC_UNWEIGHTED = 1;  (* Used by libpng < 0.95 *)
  PNG_FILTER_HEURISTIC_WEIGHTED   = 2;  (* Experimental feature *)
  PNG_FILTER_HEURISTIC_LAST       = 3;  (* Not a valid value *)

var
//#ifdef PNG_WRITE_SUPPORTED
(* Set the library compression level.  Currently, valid values range from
 * 0 - 9, corresponding directly to the zlib compression levels 0 - 9
 * (0 - no compression, 9 - "maximal" compression).  Note that tests have
 * shown that zlib compression levels 3-6 usually perform as well as level 9
 * for PNG images, and do considerably fewer caclulations.  In the future,
 * these values may not correspond directly to the zlib compression levels.
 *)
  png_set_compression_level: procedure(png_ptr: png_structp; level: integer); cdecl;
  
  png_set_compression_mem_level: procedure(png_ptr: png_structp; mem_level: integer); cdecl;
  
  png_set_compression_strategy: procedure(png_ptr: png_structp; strategy: integer); cdecl;

 (* If PNG_WRITE_OPTIMIZE_CMF_SUPPORTED is defined, libpng will use a
  * smaller value of window_bits if it can do so safely.
  *)
  png_set_compression_window_bits: procedure(png_ptr: png_structp; window_bits: integer); cdecl;
  
  png_set_compression_method: procedure(png_ptr: png_structp; method: integer); cdecl;
//#endif

//#ifdef PNG_WRITE_CUSTOMIZE_ZTXT_COMPRESSION_SUPPORTED
(* Also set zlib parameters for compressing non-IDAT chunks *)
//PNG_EXPORT(222, void, png_set_text_compression_level,
//    (png_structp png_ptr, int level));

//PNG_EXPORT(223, void, png_set_text_compression_mem_level, (png_structp png_ptr,
//    int mem_level));

//PNG_EXPORT(224, void, png_set_text_compression_strategy, (png_structp png_ptr,
//    int strategy));

(* If PNG_WRITE_OPTIMIZE_CMF_SUPPORTED is defined, libpng will use a
 * smaller value of window_bits if it can do so safely.
 *)
//PNG_EXPORT(225, void, png_set_text_compression_window_bits, (png_structp
//    png_ptr, int window_bits));

//PNG_EXPORT(226, void, png_set_text_compression_method, (png_structp png_ptr,
//    int method));
//#endif (* PNG_WRITE_CUSTOMIZE_ZTXT_COMPRESSION_SUPPORTED *)

(* These next functions are called for input/output, memory, and error
 * handling.  They are in the file pngrio.c, pngwio.c, and pngerror.c,
 * and call standard C I/O routines such as fread(), fwrite(), and
 * fprintf().  These functions can be made to use other I/O routines
 * at run time for those applications that need to handle I/O in a
 * different manner by calling png_set_???_fn().  See libpng-manual.txt for
 * more information.
 *)

//#ifdef PNG_STDIO_SUPPORTED 
(* Initialize the input/output for the PNG file to the default functions. *)
  png_init_io: procedure(png_ptr: png_structp; fp: png_FILE_p);
//#endif
  
(* Replace the (error and abort), and warning functions with user
 * supplied functions.  If no messages are to be printed you must still
 * write and use replacement functions. The replacement error_fn should
 * still do a longjmp to the last setjmp location if you are using this
 * method of error handling.  If error_fn or warning_fn is NULL, the
 * default function will be used.
 *)
 
  png_set_error_fn: procedure(png_ptr: png_structp; error_ptr: png_voidp; error_fn: png_error_ptr; warning_fn: png_error_ptr); cdecl;

(* Return the user pointer associated with the error functions *)
  png_get_error_ptr: function(png_ptr: png_const_structp): png_voidp; cdecl;

(* Replace the default data output functions with a user supplied one(s).
 * If buffered output is not used, then output_flush_fn can be set to NULL.
 * If PNG_WRITE_FLUSH_SUPPORTED is not defined at libpng compile time
 * output_flush_fn will be ignored (and thus can be NULL).
 * It is probably a mistake to use NULL for output_flush_fn if
 * write_data_fn is not also NULL unless you have built libpng with
 * PNG_WRITE_FLUSH_SUPPORTED undefined, because in this case libpng's
 * default flush function, which uses the standard *FILE structure, will
 * be used.
 *)
  png_set_write_fn: procedure(png_ptr: png_structp; io_ptr: png_voidp; write_data_fn: png_rw_ptr; output_flush_fn: png_flush_ptr); cdecl;

(* Replace the default data input function with a user supplied one. *)
  png_set_read_fn: procedure(png_ptr: png_structp; io_ptr: png_voidp; read_data_fn: png_rw_ptr); cdecl;

(* Return the user pointer associated with the I/O functions *)
  png_get_io_ptr: function(png_ptr: png_structp): png_voidp; cdecl;

  png_set_read_status_fn: procedure(png_ptr: png_structp; read_row_fn: png_read_status_ptr); cdecl;
  png_set_write_status_fn: procedure(png_ptr: png_structp; write_row_fn: png_write_status_ptr); cdecl;

//#ifdef PNG_USER_MEM_SUPPORTED
(* Replace the default memory allocation functions with user supplied one(s). *)
  png_set_mem_fn: procedure(png_ptr: png_structp; mem_ptr: png_voidp; malloc_fn: png_malloc_ptr; free_fn: png_free_ptr); cdecl;
  
(* Return the user pointer associated with the memory functions *)
  png_get_mem_ptr: function(png_ptr: png_const_structp): png_voidp; cdecl;
//#endif

//#ifdef PNG_READ_USER_TRANSFORM_SUPPORTED
  png_set_read_user_transform_fn: procedure(png_ptr: png_structp; read_user_transform_fn: png_user_transform_ptr); cdecl;
//#endif

//#ifdef PNG_WRITE_USER_TRANSFORM_SUPPORTED
  png_set_write_user_transform_fn: procedure(png_ptr: png_structp; write_user_transform_fn: png_user_transform_ptr); cdecl;
//#endif

//#ifdef PNG_USER_TRANSFORM_PTR_SUPPORTED
  png_set_user_transform_info: procedure(png_ptr: png_structp; user_transform_ptr: png_voidp; user_transform_depth: integer; user_transform_channels: integer); cdecl;

(* Return the user pointer associated with the user transform functions *)
  png_get_user_transform_ptr: function(png_ptr: png_const_structp): png_voidp;
//#endif

//#ifdef PNG_USER_TRANSFORM_INFO_SUPPORTED
(* Return information about the row currently being processed.  Note that these
 * APIs do not fail but will return unexpected results if called outside a user
 * transform callback.  Also note that when transforming an interlaced image the
 * row number is the row number within the sub-image of the interlace pass, so
 * the value will increase to the height of the sub-image (not the full image)
 * then reset to 0 for the next pass.
 *
 * Use PNG_ROW_FROM_PASS_ROW(row, pass) and PNG_COL_FROM_PASS_COL(col, pass) to
 * find the output pixel (x,y) given an interlaced sub-image pixel
 * (row,col,pass).  (See below for these macros.)
 *)
  png_get_current_row_number: function(png_ptr: png_const_structp): png_uint_32; cdecl;
  png_get_current_pass_number: function(png_ptr: png_const_structp): png_byte; cdecl;
//#endif

//#ifdef PNG_USER_CHUNKS_SUPPORTED
  png_set_read_user_chunk_fn: procedure(png_ptr: png_structp; user_chunk_ptr: png_voidp; read_user_chunk_fn: png_user_chunk_ptr); cdecl;
  png_get_user_chunk_ptr: function(png_ptr: png_const_structp): png_voidp;
//#endif

//#ifdef PNG_PROGRESSIVE_READ_SUPPORTED
(* Sets the function callbacks for the push reader, and a pointer to a
 * user-defined structure available to the callback functions.
 *)
  png_set_progressive_read_fn: procedure (png_ptr: png_structp; progressive_ptr: png_voidp; info_fn: png_progressive_info_ptr; row_fn: png_progressive_row_ptr; end_fn: png_progressive_end_ptr); cdecl;

(* returns the user pointer associated with the push read functions *)
  png_get_progressive_ptr: function(png_ptr: png_const_structp): png_voidp; cdecl;

(* function to be called when data becomes available *)
  png_process_data: procedure(png_ptr: png_structp; info_ptr: png_infop; buffer: png_bytep; buffer_size: png_size_t); cdecl;
 
(* A function which may be called *only* within png_process_data to stop the
 * processing of any more data.  The function returns the number of bytes
 * remaining, excluding any that libpng has cached internally.  A subsequent
 * call to png_process_data must supply these bytes again.  If the argument
 * 'save' is set to true the routine will first save all the pending data and
 * will always return 0.
 *)
  png_process_data_pause: function(png_ptr: png_structp; save: integer): png_size_t; cdecl;

(* A function which may be called *only* outside (after) a call to
 * png_process_data.  It returns the number of bytes of data to skip in the
 * input.  Normally it will return 0, but if it returns a non-zero value the
 * application must skip than number of bytes of input data and pass the
 * following data to the next call to png_process_data.
 *)
  png_process_data_skip: function(png_ptr: png_structp): png_uint_32; cdecl;

//#ifdef PNG_READ_INTERLACING_SUPPORTED
(* Function that combines rows.  'new_row' is a flag that should come from
 * the callback and be non-NULL if anything needs to be done; the library
 * stores its own version of the new data internally and ignores the passed
 * in value.
 *)
  png_progressive_combine_row: procedure(png_ptr: png_structp; old_row: png_bytep; new_row: png_const_bytep); cdecl;
//#endif (* PNG_READ_INTERLACING_SUPPORTED *)
//#endif (* PNG_PROGRESSIVE_READ_SUPPORTED *)

  png_malloc: function(png_ptr: png_structp; size: png_alloc_size_t): png_voidp; cdecl;
  
(* Added at libpng version 1.4.0 *)
  png_calloc: function(png_ptr: png_structp; size: png_alloc_size_t): png_voidp; cdecl;

(* Added at libpng version 1.2.4 *)
  png_malloc_warn: function(png_ptr: png_structp; size: png_alloc_size_t): png_voidp; cdecl;

(* frees a pointer allocated by png_malloc() *)
  png_free: procedure(png_ptr: png_structp; ptr: png_voidp); cdecl;

(* Free data that was allocated internally *)
  png_free_data: procedure(png_ptr: png_structp; info_ptr: png_infop; free_me: png_uint_32; num: integer); cdecl;

(* Reassign responsibility for freeing existing data, whether allocated
  by libpng or by the application *)
  png_data_freer: procedure(png_ptr: png_structp; info_ptr: png_infop; freer: integer; mask: png_uint_32); cdecl;

const
(* assignments for png_data_freer *)
  PNG_DESTROY_WILL_FREE_DATA = 1;
  PNG_SET_WILL_FREE_DATA = 1;
  PNG_USER_WILL_FREE_DATA = 2;
(* Flags for png_ptr->free_me and info_ptr->free_me *)
  PNG_FREE_HIST = $0008;
  PNG_FREE_ICCP = $0010;
  PNG_FREE_SPLT = $0020;
  PNG_FREE_ROWS = $0040;
  PNG_FREE_PCAL = $0080;
  PNG_FREE_SCAL = $0100;
  PNG_FREE_UNKN = $0200;
  PNG_FREE_LIST = $0400;
  PNG_FREE_PLTE = $1000;
  PNG_FREE_TRNS = $2000;
  PNG_FREE_TEXT = $4000;
  PNG_FREE_ALL  = $7fff;
  PNG_FREE_MUL  = $4220; (* PNG_FREE_SPLT|PNG_FREE_TEXT|PNG_FREE_UNKN *)
  
var 
//#ifdef PNG_USER_MEM_SUPPORTED 
  png_malloc_default: function(png_ptr: png_structp; size: png_alloc_size_t): png_voidp; cdecl;
  png_free_default: procedure(png_ptr: png_structp; ptr: png_voidp); cdecl;
//#endif

//#ifdef PNG_ERROR_TEXT_SUPPORTED
(* Fatal error in PNG image of libpng - can't continue *)
  png_error: procedure(png_ptr: png_structp; error_message: png_const_charp); cdecl;

(* The same, but the chunk name is prepended to the error string. *)
  png_chunk_error: procedure(png_ptr: png_structp; error_message: png_const_charp); cdecl;
//#else
(* Fatal error in PNG image of libpng - can't continue *)
  png_err: procedure(png_ptr: png_structp); cdecl;
//#endif

//#ifdef PNG_WARNINGS_SUPPORTED
(* Non-fatal error in libpng.  Can continue, but may have a problem. *)
  png_warning: procedure(png_ptr: png_structp; warning_message: png_const_charp); cdecl;

(* Non-fatal error in libpng, chunk name is prepended to message. *)
  png_chunk_warning: procedure(png_ptr: png_structp; warning_message: png_const_charp); cdecl;
//#endif

//#ifdef PNG_BENIGN_ERRORS_SUPPORTED
(* Benign error in libpng.  Can continue, but may have a problem.
 * User can choose whether to handle as a fatal error or as a warning. *)
//#  undef png_benign_error
//PNG_EXPORT(107, void, png_benign_error, (png_structp png_ptr,
//    png_const_charp warning_message));

(* Same, chunk name is prepended to message. *)
//#  undef png_chunk_benign_error
//PNG_EXPORT(108, void, png_chunk_benign_error, (png_structp png_ptr,
    //png_const_charp warning_message));

//PNG_EXPORT(109, void, png_set_benign_errors,
    //(png_structp png_ptr, int allowed));
//#else
//#  ifdef PNG_ALLOW_BENIGN_ERRORS
//#    define png_benign_error png_warning
//#    define png_chunk_benign_error png_chunk_warning
//#  else
//#    define png_benign_error png_error
//#    define png_chunk_benign_error png_chunk_error
//#  endif
//#endif

(* The png_set_<chunk> functions are for storing values in the png_info_struct.
  Similarly, the png_get_<chunk> calls are used to read values from the
  png_info_struct, either storing the parameters in the passed variables, or
  setting pointers into the png_info_struct where the data is stored.  The
  png_get_<chunk> functions return a non-zero value if the data was available
  in info_ptr, or return zero and do not change any of the parameters if the
  data was not available.
 
  These functions should be used instead of directly accessing png_info
  to avoid problems with future changes in the size and internal layout of
  png_info_struct. *)
  
(* Returns "flag" if chunk data is valid in info_ptr. *)
  png_get_valid: function(png_ptr: png_const_structp; info_ptr: png_const_infop; flag: png_uint_32): png_uint_32; cdecl;

(* Returns number of bytes needed to hold a transformed row. *)
  png_get_rowbytes: function(png_ptr: png_const_structp; info_ptr: png_const_infop): png_size_t; cdecl;

(* Returns row_pointers, which is an array of pointers to scanlines that was
  returned from png_read_png(). *)
  png_get_rows: function(png_ptr: png_structp; info_ptr: png_infop): png_bytepp; cdecl;
  
(* Set row_pointers, which is an array of pointers to scanlines for use
  by png_write_png(). *)
  png_set_rows: procedure(png_ptr: png_structp; info_ptr: png_infop; row_pointers: png_bytepp); cdecl;

(* Returns number of color channels in image. *)
  png_get_channels: function(png_ptr: png_structp; info_ptr: png_infop): png_byte; cdecl;

(* Returns image width in pixels. *)
  png_get_image_width: function(png_ptr: png_structp; info_ptr: png_infop): png_uint_32; cdecl;

(* Returns image height in pixels. *)
  png_get_image_height: function(png_ptr: png_structp; info_ptr: png_infop): png_uint_32; cdecl;

(* Returns image bit_depth. *)
  png_get_bit_depth: function(png_ptr: png_structp; info_ptr: png_infop): png_byte; cdecl;

(* Returns image color_type. *)
  png_get_color_type: function(png_ptr: png_structp; info_ptr: png_infop): png_byte; cdecl;

(* Returns image filter_type. *)
  png_get_filter_type: function(png_ptr: png_structp; info_ptr: png_infop): png_byte; cdecl;

(* Returns image interlace_type. *)
  png_get_interlace_type: function(png_ptr: png_structp; info_ptr: png_infop): png_byte; cdecl;

(* Returns image compression_type. *)
  png_get_compression_type: function(png_ptr: png_structp; info_ptr: png_infop): png_byte; cdecl;
  
(* Returns image resolution in pixels per meter, from pHYs chunk data. *)
  png_get_pixels_per_meter: function(png_ptr: png_structp; info_ptr: png_infop): png_uint_32; cdecl;
  png_get_x_pixels_per_meter: function(png_ptr: png_structp; info_ptr: png_infop): png_uint_32; cdecl;
  png_get_y_pixels_per_meter: function(png_ptr: png_structp; info_ptr: png_infop): png_uint_32; cdecl;

(* Returns pixel aspect ratio, computed from pHYs chunk data. *)
  png_get_pixel_aspect_ratio: function(png_ptr: png_structp; info_ptr: png_infop): single; cdecl;

(* Returns image x, y offset in pixels or microns, from oFFs chunk data. *)
  png_get_x_offset_pixels: function(png_ptr: png_structp; info_ptr: png_infop): png_int_32; cdecl;
  png_get_y_offset_pixels: function(png_ptr: png_structp; info_ptr: png_infop): png_int_32; cdecl;
  png_get_x_offset_microns: function(png_ptr: png_structp; info_ptr: png_infop): png_int_32; cdecl;
  png_get_y_offset_microns: function(png_ptr: png_structp; info_ptr: png_infop): png_int_32; cdecl;

(* Returns pointer to signature string read from PNG header *)
  png_get_signature: function(png_ptr: png_structp; info_ptr: png_infop): png_bytep; cdecl;

  
  png_get_bKGD: function(png_ptr: png_structp; info_ptr: png_infop; background: png_color_16pp): png_uint_32; cdecl;
  png_set_bKGD: procedure(png_ptr: png_structp; info_ptr: png_infop; background: png_color_16p); cdecl;

  png_get_cHRM: function(png_ptr: png_structp; info_ptr: png_infop; white_x: pdouble; white_y: pdouble; red_x: pdouble; red_y: pdouble; green_x: pdouble; green_y: pdouble; blue_x: pdouble; blue_y: pdouble): png_uint_32; cdecl;
  png_get_cHRM_fixed: function(png_ptr: png_structp; info_ptr: png_infop; int_white_x: png_fixed_pointp; int_white_y: png_fixed_pointp; int_red_x: png_fixed_pointp; int_red_y: png_fixed_pointp; int_green_x: png_fixed_pointp; int_green_y: png_fixed_pointp; int_blue_x: png_fixed_pointp; int_blue_y: png_fixed_pointp): png_uint_32; cdecl;
  png_set_cHRM: procedure(png_ptr: png_structp; info_ptr: png_infop; white_x: double; white_y: double; red_x: double; red_y: double; green_x: double; green_y: double; blue_x: double; blue_y: double); cdecl;
  png_set_cHRM_fixed: procedure(png_ptr: png_structp; info_ptr: png_infop; int_white_x: png_fixed_point; int_white_y: png_fixed_point; int_red_x: png_fixed_point; int_red_y: png_fixed_point; int_green_x: png_fixed_point; int_green_y: png_fixed_point; int_blue_x: png_fixed_point; int_blue_y: png_fixed_point); cdecl;

  png_get_gAMA: function(png_ptr: png_structp; info_ptr: png_infop; file_gamma: pdouble): png_uint_32; cdecl;
  png_get_gAMA_fixed: function(png_ptr: png_structp; info_ptr: png_infop; int_file_gamma: png_fixed_pointp): png_uint_32; cdecl;
  png_set_gAMA: procedure(png_ptr: png_structp; info_ptr: png_infop; file_gamma: double); cdecl;
  png_set_gAMA_fixed: procedure(png_ptr: png_structp; info_ptr: png_infop; int_file_gamma: png_fixed_point); cdecl;

  png_get_hIST: function(png_ptr: png_structp; info_ptr: png_infop; hist: png_uint_16pp): png_uint_32; cdecl;
  png_set_hIST: procedure(png_ptr: png_structp; info_ptr: png_infop; hist: png_uint_16p); cdecl;
  
  png_get_IHDR: function(png_ptr: png_structp; info_ptr: png_infop; width: png_uint_32p; height: png_uint_32p; bit_depth: pinteger; color_type: pinteger; interlace_method: pinteger; compression_method: pinteger; filter_method: pinteger): png_uint_32; cdecl;
  png_set_IHDR: procedure(png_ptr: png_structp; info_ptr: png_infop; width: png_uint_32; height: png_uint_32; bit_depth: integer; color_type: integer; interlace_method: integer; compression_method: integer; filter_method: integer); cdecl;
  
  png_get_oFFs: function(png_ptr: png_structp; info_ptr: png_infop; offset_x: png_int_32p; offset_y: png_int_32p; unit_type: pinteger): png_uint_32; cdecl;
  png_set_oFFs: procedure(png_ptr: png_structp; info_ptr: png_infop; offset_x: png_int_32; offset_y: png_int_32; unit_type: integer); cdecl;

  png_get_pCAL: function(png_ptr: png_structp; info_ptr: png_infop; purpose: png_charpp; X0: png_int_32p; X1: png_int_32p; _type: pinteger; nparams: pinteger; units: png_charpp; params: png_charppp): png_uint_32; cdecl;
  png_set_pCAL: procedure(png_ptr: png_structp; info_ptr: png_infop; purpose: png_charp; X0: png_int_32; X1: png_int_32; _type: integer; nparams: integer; units: png_charp; params: png_charpp); cdecl;
  
  png_get_pHYs: function(png_ptr: png_structp; info_ptr: png_infop; res_x: png_uint_32p; res_y: png_uint_32p; unit_type: pinteger): png_uint_32; cdecl;
  png_set_pHYs: procedure(png_ptr: png_structp; info_ptr: png_infop; res_x: png_uint_32; res_y: png_uint_32; unit_type: integer); cdecl;

  png_get_PLTE: function(png_ptr: png_structp; info_ptr: png_infop; palette: png_colorpp; num_palette: pinteger): png_uint_32; cdecl;
  png_set_PLTE: procedure(png_ptr: png_structp; info_ptr: png_infop; palette: png_colorp; num_palette: integer) cdecl;

  png_get_sBIT: function(png_ptr: png_structp; info_ptr: png_infop; sig_bit: png_color_8pp): png_uint_32; cdecl;
  png_set_sBIT: procedure(png_ptr: png_structp; info_ptr: png_infop; sig_bit: png_color_8p); cdecl;
  
  png_get_sRGB: function(png_ptr: png_structp; info_ptr: png_infop; intent: pinteger): png_uint_32; cdecl;
  png_set_sRGB: procedure(png_ptr: png_structp; info_ptr: png_infop; intent: integer); cdecl;
  png_set_sRGB_gAMA_and_cHRM: procedure(png_ptr: png_structp; info_ptr: png_infop; intent: integer); cdecl;
  
  png_get_iCCP: function(png_ptr: png_structp; info_ptr: png_infop; name: png_charpp; compression_type: pinteger; profile: png_charpp; proflen: png_uint_32p): png_uint_32; cdecl;
(* Note to maintainer: profile should be png_bytepp *)

  png_set_iCCP: procedure(png_ptr: png_structp; info_ptr: png_infop; name: png_charp; compression_type: integer; profile: png_charp; proflen: png_uint_32); cdecl;
(* Note to maintainer: profile should be png_bytep *)

  png_get_sPLT: function(png_ptr: png_structp; info_ptr: png_infop; entries: png_sPLT_tpp): png_uint_32; cdecl;
  png_set_sPLT: procedure(png_ptr: png_structp; info_ptr: png_infop; entries: png_sPLT_tp; nentries: integer); cdecl;
  
(* png_get_text also returns the number of text chunks in *num_text *)
  png_get_text: function(png_ptr: png_structp; info_ptr: png_infop; text_ptr: png_textpp; num_text: pinteger): png_uint_32; cdecl;
  
(* Note while png_set_text() will accept a structure whose text,
  language, and  translated keywords are NULL pointers, the structure
  returned by png_get_text will always contain regular
  zero-terminated C strings.  They might be empty strings but
  they will never be NULL pointers. *)
  png_set_text: procedure(png_ptr: png_structp; info_ptr: png_infop; text_ptr: png_textp; num_text: integer); cdecl;
  
  png_get_tIME: function(png_ptr: png_structp; info_ptr: png_infop; mod_time: png_timepp): png_uint_32; cdecl;
  png_set_tIME: procedure(png_ptr: png_structp; info_ptr: png_infop; mod_time: png_timep); cdecl;
  
  png_get_tRNS: function(png_ptr: png_structp; info_ptr: png_infop; trans: png_bytepp; num_trans: pinteger; trans_values: png_color_16pp): png_uint_32; cdecl;
  png_set_tRNS: procedure(png_ptr: png_structp; info_ptr: png_infop; trans: png_bytep; num_trans: integer; trans_values: png_color_16p); cdecl;

  png_get_sCAL: function(png_ptr: png_structp; info_ptr: png_infop; _unit: pinteger; width: pdouble; height: pdouble): png_uint_32; cdecl;
  png_get_sCAL_s: function(png_ptr: png_structp; info_ptr: png_infop; _unit: pinteger; swidth: png_charpp; sheight: png_charpp): png_uint_32; cdecl;
  png_set_sCAL: procedure(png_ptr: png_structp; info_ptr: png_infop; _unit: integer; width: double; height: double); cdecl;
  png_set_sCAL_s: procedure(png_ptr: png_structp; info_ptr: png_infop; _unit: integer; swidth: png_charp; sheight: png_charp); cdecl;
  
(* provide a list of chunks and how they are to be handled, if the built-in
  handling or default unknown chunk handling is not desired.  Any chunks not
  listed will be handled in the default manner.  The IHDR and IEND chunks
  must not be listed.
     keep = 0: follow default behavour
          = 1: do not keep
          = 2: keep only if safe-to-copy
          = 3: keep even if unsafe-to-copy *)
  png_set_keep_unknown_chunks: procedure(png_ptr: png_structp; keep: integer; chunk_list: png_bytep; num_chunks: integer); cdecl;
  png_set_unknown_chunks: procedure(png_ptr: png_structp; info_ptr: png_infop; unknowns: png_unknown_chunkp; num_unknowns: integer); cdecl;
  png_set_unknown_chunk_location: procedure(png_ptr: png_structp; info_ptr: png_infop; chunk: integer; location: integer); cdecl;
  png_get_unknown_chunks: function(png_ptr: png_structp; info_ptr: png_infop; entries: png_unknown_chunkpp): png_uint_32; cdecl;

  png_handle_as_unknown: function(png_ptr: png_structp; chunk_name: png_bytep): integer; cdecl;
  
 
(* Png_free_data() will turn off the "valid" flag for anything it frees.
  If you need to turn it off for a chunk that your application has freed,
  you can use png_set_invalid(png_ptr, info_ptr, PNG_INFO_CHNK); *)
  png_set_invalid: procedure(png_ptr: png_structp; info_ptr: png_infop; mask: integer); cdecl;

(* The "params" pointer is currently not used and is for future expansion. *)
  png_read_png: procedure(png_ptr: png_structp; info_ptr: png_infop; transforms: integer; params: png_voidp); cdecl;
  png_write_png: procedure(png_ptr: png_structp; info_ptr: png_infop; transforms: integer; params: png_voidp); cdecl;
  

  png_sig_bytes: function: png_bytep; cdecl;

  png_get_copyright: function(png_ptr: png_structp): png_charp; cdecl;
  png_get_header_ver: function(png_ptr: png_structp): png_charp; cdecl;
  png_get_header_version: function(png_ptr: png_structp): png_charp; cdecl;
  png_get_libpng_ver: function(png_ptr: png_structp): png_charp; cdecl;

  png_permit_mng_features: function(png_ptr: png_structp; mng_features_permitted: png_uint_32): png_uint_32; cdecl;

const  
  PNG_HANDLE_CHUNK_AS_DEFAULT   = 0;
  PNG_HANDLE_CHUNK_NEVER        = 1;
  PNG_HANDLE_CHUNK_IF_SAFE      = 2;
  PNG_HANDLE_CHUNK_ALWAYS       = 3;

(* Added to version 1.2.0 *)
  PNG_ASM_FLAG_MMX_SUPPORT_COMPILED  = $01;  (* not user-settable *)
  PNG_ASM_FLAG_MMX_SUPPORT_IN_CPU    = $02;  (* not user-settable *)
  PNG_ASM_FLAG_MMX_READ_COMBINE_ROW  = $04;
  PNG_ASM_FLAG_MMX_READ_INTERLACE    = $08;
  PNG_ASM_FLAG_MMX_READ_FILTER_SUB   = $10;
  PNG_ASM_FLAG_MMX_READ_FILTER_UP    = $20;
  PNG_ASM_FLAG_MMX_READ_FILTER_AVG   = $40;
  PNG_ASM_FLAG_MMX_READ_FILTER_PAETH = $80;
  PNG_ASM_FLAGS_INITIALIZED          = $80000000;  (* not user-settable *)

  PNG_MMX_READ_FLAGS = PNG_ASM_FLAG_MMX_READ_COMBINE_ROW or
                       PNG_ASM_FLAG_MMX_READ_INTERLACE   or
                       PNG_ASM_FLAG_MMX_READ_FILTER_SUB  or
                       PNG_ASM_FLAG_MMX_READ_FILTER_UP   or
                       PNG_ASM_FLAG_MMX_READ_FILTER_AVG  or
                       PNG_ASM_FLAG_MMX_READ_FILTER_PAETH;
  PNG_MMX_WRITE_FLAGS = 0;

  PNG_MMX_FLAGS = PNG_ASM_FLAG_MMX_SUPPORT_COMPILED or
                  PNG_ASM_FLAG_MMX_SUPPORT_IN_CPU   or
                  PNG_MMX_READ_FLAGS                or
                  PNG_MMX_WRITE_FLAGS;

  PNG_SELECT_READ   = 1;
  PNG_SELECT_WRITE  = 2;
  

var
  png_get_mmx_flagmask: function(flag_select: integer; compilerID: pinteger): png_uint_32; cdecl;
  png_get_asm_flagmask: function(flag_select: integer): png_uint_32; cdecl;
  png_get_asm_flags: function(png_ptr: png_structp): png_uint_32; cdecl;
  png_get_mmx_bitdepth_threshold: function(png_ptr: png_structp): png_byte; cdecl;
  png_get_mmx_rowbytes_threshold: function(png_ptr: png_structp): png_uint_32; cdecl;
  png_set_asm_flags: procedure(png_ptr: png_structp; asm_flags: png_uint_32); cdecl;
  png_set_mmx_thresholds: procedure(png_ptr: png_structp; mmx_bitdepth_threshold: png_byte; mmx_rowbytes_threshold: png_uint_32); cdecl;
  
  png_mmx_support: function: integer; cdecl;
  
(* Strip the prepended error numbers ("#nnn ") from error and warning
  messages before passing them to the error or warning handler. *)
  png_set_strip_error_numbers: procedure(png_ptr: png_structp; strip_mode: png_uint_32); cdecl;
  
  png_set_user_limits: procedure(png_ptr: png_structp; user_width_max: png_uint_32; user_height_max: png_uint_32); cdecl;
  png_get_user_width_max: function(png_ptr: png_structp): png_uint_32; cdecl;
  png_get_user_height_max: function(png_ptr: png_structp): png_uint_32; cdecl;

(* Inline macros to do direct reads of bytes from the input buffer.  These
  require that you are using an architecture that uses PNG byte ordering
  (MSB first) and supports unaligned data storage.  I think that PowerPC
  in big-endian mode and 680x0 are the only ones that will support this.
  The x86 line of processors definitely do not.  The png_get_int_32()
  routine also assumes we are using two's complement format for negative
  values, which is almost certainly true. *)
  png_get_uint_32: function(buf: png_bytep): png_uint_32; cdecl;
  png_get_uint_16: function(buf: png_bytep): png_uint_16; cdecl;
  png_get_int_32: function (buf: png_bytep): png_int_32; cdecl;
  png_get_uint_31: function(png_ptr: png_structp; buf: png_bytep): png_uint_32; cdecl;
(* No png_get_int_16 -- may be added if there's a real need for it. *)
  
(* Place a 32-bit number into a buffer in PNG byte order (big-endian). *)
  png_save_uint_32: procedure(buf: png_bytep; i: png_uint_32); cdecl;
  png_save_int_32: procedure(buf: png_bytep; i: png_int_32); cdecl;

(* Place a 16-bit number into a buffer in PNG byte order.
  The parameter is declared unsigned int, not png_uint_16,
  just to avoid potential problems on pre-ANSI C compilers. *)
  png_save_uint_16: procedure(buf: png_bytep; i: word); cdecl;
(* No png_save_int_16 -- may be added if there's a real need for it. *)


(* Initialize png_ptr struct for reading, and allocate any other memory.
  (old interface - DEPRECATED - use png_create_read_struct instead). *)
  png_read_init: procedure(png_ptr: png_structp); cdecl;

  png_read_init_3: procedure(ptr_ptr: png_structpp; user_png_ver: png_const_charp; png_struct_size: png_size_t); cdecl;
  png_read_init_2: procedure(png_ptr: png_structp; user_png_ver: png_const_charp; png_struct_size: png_size_t; png_info_size: png_size_t); cdecl;

(* Initialize png_ptr struct for writing, and allocate any other memory.
  (old interface - DEPRECATED - use png_create_write_struct instead). *)
  png_write_init: procedure(png_ptr: png_structp); cdecl;

  png_write_init_3: procedure(ptr_ptr: png_structpp; user_png_ver: png_const_charp; png_struct_size: png_size_t); cdecl;
  png_write_init_2: procedure(png_ptr: png_structp; user_png_ver: png_const_charp; png_struct_size: png_size_t; png_info_size: png_size_t); cdecl;


  function init_libPNG(libPNG_Name: AnsiString = LIB_PNG_NAME): boolean;
  procedure quit_libPNG;


implementation


var
  libPNG_RefCount: Integer;

  (*$ifdef win32*)
    libPNG_Handle: cardinal;
  (*$else*)
    libPNG_Handle: pointer;
  (*$endif*)


(*$ifdef win32*)
const
  Kernel32 = 'kernel32.dll';

  function LoadLibrary(lpFileName: pAnsiChar): LongWord; stdcall; external Kernel32 name 'LoadLibraryA';
  function FreeLibrary(hModule: LongWord): LongBool; stdcall; external Kernel32 name 'FreeLibrary';
  function GetProcAddress(hModule: LongWord; lpProcName: pAnsiChar): Pointer; stdcall; external Kernel32 name 'GetProcAddress';
(*$else*)
const
  libdl = (*$IFDEF Linux*) 'libdl.so.2'(*$ELSE*) 'c'(*$ENDIF*);

  RTLD_LAZY = $001;

  function dlopen(Name: pAnsiChar; Flags: LongInt): Pointer; cdecl; external libdl name 'dlopen';
  function dlclose(Lib: Pointer): LongInt; cdecl; external libdl name 'dlclose';
  function dlsym(Lib: Pointer; Name: pAnsiChar): Pointer; cdecl; external libdl name 'dlsym';
(*$endif*)


function GetProcAddr(Name: pAnsiChar): Pointer;
begin
  (*$ifdef win32*)
    GetProcAddr := GetProcAddress(libPNG_Handle, Name);
  (*$else*)
    GetProcAddr := dlsym(libPNG_Handle, Name);
  (*$endif*)
end;


function init_libPNG(libPNG_Name: AnsiString): boolean;
var
  Temp: Boolean;
begin
  if (libPNG_RefCount = 0) or (libPNG_Handle = (*$ifdef win32*) 0 (*$else*) nil (*$endif*)) then begin
    if libPNG_Handle = (*$ifdef win32*) 0 (*$else*) nil (*$endif*) then
      (*$ifdef win32*)
        libPNG_Handle := LoadLibrary(pAnsiChar(libPNG_Name));
      (*$else*)
        libPNG_Handle := dlopen(pAnsiChar(libPNG_Name), RTLD_LAZY);
      (*$endif*)

    // load function pointers
    if libPNG_Handle <> (*$ifdef win32*) 0 (*$else*) nil (*$endif*) then begin
      png_access_version_number := GetProcAddr('png_access_version_number');
      png_set_sig_bytes := GetProcAddr('png_set_sig_bytes');
      png_sig_cmp := GetProcAddr('png_sig_cmp');
      png_check_sig := GetProcAddr('png_check_sig');
      png_create_read_struct := GetProcAddr('png_create_read_struct');
      png_create_write_struct := GetProcAddr('png_create_write_struct');
      png_get_compression_buffer_size := GetProcAddr('png_get_compression_buffer_size');
      png_set_compression_buffer_size := GetProcAddr('png_set_compression_buffer_size');
      png_reset_zstream := GetProcAddr('png_reset_zstream');
      png_create_read_struct_2 := GetProcAddr('png_create_read_struct_2');
      png_create_write_struct_2 := GetProcAddr('png_create_write_struct_2');
      png_write_chunk := GetProcAddr('png_write_chunk');
      png_write_chunk_start := GetProcAddr('png_write_chunk_start');
      png_write_chunk_data := GetProcAddr('png_write_chunk_data');
      png_write_chunk_end := GetProcAddr('png_write_chunk_end');
      png_create_info_struct := GetProcAddr('png_create_info_struct');
      png_info_init := GetProcAddr('png_info_init');
      png_info_init_3 := GetProcAddr('png_info_init_3');
      png_write_info_before_PLTE := GetProcAddr('png_write_info_before_PLTE');
      png_write_info := GetProcAddr('png_write_info');
      png_read_info := GetProcAddr('png_read_info');
      png_convert_to_rfc1123 := GetProcAddr('png_convert_to_rfc1123');
      png_convert_from_struct_tm := GetProcAddr('png_convert_from_struct_tm');
      png_convert_from_time_t := GetProcAddr('png_convert_from_time_t');
      png_set_expand := GetProcAddr('png_set_expand');
      png_set_expand_gray_1_2_4_to_8 := GetProcAddr('png_set_expand_gray_1_2_4_to_8');
      png_set_palette_to_rgb := GetProcAddr('png_set_palette_to_rgb');
      png_set_tRNS_to_alpha := GetProcAddr('png_set_tRNS_to_alpha');
      png_set_gray_1_2_4_to_8 := GetProcAddr('png_set_gray_1_2_4_to_8');
      png_set_bgr := GetProcAddr('png_set_bgr');
      png_set_gray_to_rgb := GetProcAddr('png_set_gray_to_rgb');
      png_set_rgb_to_gray := GetProcAddr('png_set_rgb_to_gray');
      png_set_rgb_to_gray_fixed := GetProcAddr('png_set_rgb_to_gray_fixed');
      png_get_rgb_to_gray_status := GetProcAddr('png_get_rgb_to_gray_status');
      png_build_grayscale_palette := GetProcAddr('png_build_grayscale_palette');
      png_set_strip_alpha := GetProcAddr('png_set_strip_alpha');
      png_set_swap_alpha := GetProcAddr('png_set_swap_alpha');
      png_set_invert_alpha := GetProcAddr('png_set_invert_alpha');
      png_set_filler := GetProcAddr('png_set_filler');
      png_set_add_alpha := GetProcAddr('png_set_add_alpha');
      png_set_swap := GetProcAddr('png_set_swap');
      png_set_packing := GetProcAddr('png_set_packing');
      png_set_packswap := GetProcAddr('png_set_packswap');
      png_set_shift := GetProcAddr('png_set_shift');
      png_set_interlace_handling := GetProcAddr('png_set_interlace_handling');
      png_set_invert_mono := GetProcAddr('png_set_invert_mono');
      png_set_background := GetProcAddr('png_set_background');
      png_set_strip_16 := GetProcAddr('png_set_strip_16');
      png_set_dither := GetProcAddr('png_set_dither');
      png_set_gamma := GetProcAddr('png_set_gamma');
      png_permit_empty_plte := GetProcAddr('png_permit_empty_plte');
      png_set_flush := GetProcAddr('png_set_flush');
      png_write_flush := GetProcAddr('png_write_flush');
      png_start_read_image := GetProcAddr('png_start_read_image');
      png_read_update_info := GetProcAddr('png_read_update_info');
      png_read_rows := GetProcAddr('png_read_rows');
      png_read_row := GetProcAddr('png_read_row');
      png_read_image := GetProcAddr('png_read_image');
      png_write_row := GetProcAddr('png_write_row');
      png_write_rows := GetProcAddr('png_write_rows');
      png_write_image := GetProcAddr('png_write_image');
      png_write_end := GetProcAddr('png_write_end');
      png_read_end := GetProcAddr('png_read_end');
      png_destroy_info_struct := GetProcAddr('png_destroy_info_struct');
      png_destroy_read_struct := GetProcAddr('png_destroy_read_struct');
      png_destroy_write_struct := GetProcAddr('png_destroy_write_struct');
      png_set_crc_action := GetProcAddr('png_set_crc_action');
      png_set_filter := GetProcAddr('png_set_filter');
      png_set_filter_heuristics := GetProcAddr('png_set_filter_heuristics');
      png_set_compression_level := GetProcAddr('png_set_compression_level');
      png_set_compression_mem_level := GetProcAddr('png_set_compression_mem_level');
      png_set_compression_strategy := GetProcAddr('png_set_compression_strategy');
      png_set_compression_window_bits := GetProcAddr('png_set_compression_window_bits');
      png_set_compression_method := GetProcAddr('png_set_compression_method');
      png_init_io := GetProcAddr('png_init_io');
      png_set_error_fn := GetProcAddr('png_set_error_fn');
      png_get_error_ptr := GetProcAddr('png_get_error_ptr');
      png_set_write_fn := GetProcAddr('png_set_write_fn');
      png_set_read_fn := GetProcAddr('png_set_read_fn');
      png_get_io_ptr := GetProcAddr('png_get_io_ptr');
      png_set_read_status_fn := GetProcAddr('png_set_read_status_fn');
      png_set_write_status_fn := GetProcAddr('png_set_write_status_fn');
      png_set_mem_fn := GetProcAddr('png_set_mem_fn');
      png_get_mem_ptr := GetProcAddr('png_get_mem_ptr');
      png_set_read_user_transform_fn := GetProcAddr('png_set_read_user_transform_fn');
      png_set_write_user_transform_fn := GetProcAddr('png_set_write_user_transform_fn');
      png_set_user_transform_info := GetProcAddr('png_set_user_transform_info');
      png_get_user_transform_ptr := GetProcAddr('png_get_user_transform_ptr');
      png_set_read_user_chunk_fn := GetProcAddr('png_set_read_user_chunk_fn');
      png_get_user_chunk_ptr := GetProcAddr('png_get_user_chunk_ptr');
      png_set_progressive_read_fn := GetProcAddr('png_set_progressive_read_fn');
      png_get_progressive_ptr := GetProcAddr('png_get_progressive_ptr');
      png_process_data := GetProcAddr('png_process_data');
      png_progressive_combine_row := GetProcAddr('png_progressive_combine_row');
      png_malloc := GetProcAddr('png_malloc');
      png_malloc_warn := GetProcAddr('png_malloc_warn');
      png_free := GetProcAddr('png_free');
      png_zalloc := GetProcAddr('png_zalloc');
      png_zfree := GetProcAddr('png_zfree');
      png_free_data := GetProcAddr('png_free_data');
      png_data_freer := GetProcAddr('png_data_freer');
      png_malloc_default := GetProcAddr('png_malloc_default');
      png_free_default := GetProcAddr('png_free_default');
      png_memcpy_check := GetProcAddr('png_memcpy_check');
      png_memset_check := GetProcAddr('png_memset_check');
      png_far_to_near := GetProcAddr('png_far_to_near');
      png_error := GetProcAddr('png_error');
      png_chunk_error := GetProcAddr('png_chunk_error');
      png_warning := GetProcAddr('png_warning');
      png_chunk_warning := GetProcAddr('png_chunk_warning');
      png_get_valid := GetProcAddr('png_get_valid');
      png_get_rowbytes := GetProcAddr('png_get_rowbytes');
      png_get_rows := GetProcAddr('png_get_rows');
      png_set_rows := GetProcAddr('png_set_rows');
      png_get_channels := GetProcAddr('png_get_channels');
      png_get_image_width := GetProcAddr('png_get_image_width');
      png_get_image_height := GetProcAddr('png_get_image_height');
      png_get_bit_depth := GetProcAddr('png_get_bit_depth');
      png_get_color_type := GetProcAddr('png_get_color_type');
      png_get_filter_type := GetProcAddr('png_get_filter_type');
      png_get_interlace_type := GetProcAddr('png_get_interlace_type');
      png_get_compression_type := GetProcAddr('png_get_compression_type');
      png_get_pixels_per_meter := GetProcAddr('png_get_pixels_per_meter');
      png_get_x_pixels_per_meter := GetProcAddr('png_get_x_pixels_per_meter');
      png_get_y_pixels_per_meter := GetProcAddr('png_get_y_pixels_per_meter');
      png_get_pixel_aspect_ratio := GetProcAddr('png_get_pixel_aspect_ratio');
      png_get_x_offset_pixels := GetProcAddr('png_get_x_offset_pixels');
      png_get_y_offset_pixels := GetProcAddr('png_get_y_offset_pixels');
      png_get_x_offset_microns := GetProcAddr('png_get_x_offset_microns');
      png_get_y_offset_microns := GetProcAddr('png_get_y_offset_microns');
      png_get_signature := GetProcAddr('png_get_signature');
      png_get_bKGD := GetProcAddr('png_get_bKGD');
      png_set_bKGD := GetProcAddr('png_set_bKGD');
      png_get_cHRM := GetProcAddr('png_get_cHRM');
      png_get_cHRM_fixed := GetProcAddr('png_get_cHRM_fixed');
      png_set_cHRM := GetProcAddr('png_set_cHRM');
      png_set_cHRM_fixed := GetProcAddr('png_set_cHRM_fixed');
      png_get_gAMA := GetProcAddr('png_get_gAMA');
      png_get_gAMA_fixed := GetProcAddr('png_get_gAMA_fixed');
      png_set_gAMA := GetProcAddr('png_set_gAMA');
      png_set_gAMA_fixed := GetProcAddr('png_set_gAMA_fixed');
      png_get_hIST := GetProcAddr('png_get_hIST');
      png_set_hIST := GetProcAddr('png_set_hIST');
      png_get_IHDR := GetProcAddr('png_get_IHDR');
      png_set_IHDR := GetProcAddr('png_set_IHDR');
      png_get_oFFs := GetProcAddr('png_get_oFFs');
      png_set_oFFs := GetProcAddr('png_set_oFFs');
      png_get_pCAL := GetProcAddr('png_get_pCAL');
      png_set_pCAL := GetProcAddr('png_set_pCAL');
      png_get_pHYs := GetProcAddr('png_get_pHYs');
      png_set_pHYs := GetProcAddr('png_set_pHYs');
      png_get_PLTE := GetProcAddr('png_get_PLTE');
      png_set_PLTE := GetProcAddr('png_set_PLTE');
      png_get_sBIT := GetProcAddr('png_get_sBIT');
      png_set_sBIT := GetProcAddr('png_set_sBIT');
      png_get_sRGB := GetProcAddr('png_get_sRGB');
      png_set_sRGB := GetProcAddr('png_set_sRGB');
      png_set_sRGB_gAMA_and_cHRM := GetProcAddr('png_set_sRGB_gAMA_and_cHRM');
      png_get_iCCP := GetProcAddr('png_get_iCCP');
      png_set_iCCP := GetProcAddr('png_set_iCCP');
      png_get_sPLT := GetProcAddr('png_get_sPLT');
      png_set_sPLT := GetProcAddr('png_set_sPLT');
      png_get_text := GetProcAddr('png_get_text');
      png_set_text := GetProcAddr('png_set_text');
      png_get_tIME := GetProcAddr('png_get_tIME');
      png_set_tIME := GetProcAddr('png_set_tIME');
      png_get_tRNS := GetProcAddr('png_get_tRNS');
      png_set_tRNS := GetProcAddr('png_set_tRNS');
      png_get_sCAL := GetProcAddr('png_get_sCAL');
      png_get_sCAL_s := GetProcAddr('png_get_sCAL_s');
      png_set_sCAL := GetProcAddr('png_set_sCAL');
      png_set_sCAL_s := GetProcAddr('png_set_sCAL_s');
      png_set_keep_unknown_chunks := GetProcAddr('png_set_keep_unknown_chunks');
      png_set_unknown_chunks := GetProcAddr('png_set_unknown_chunks');
      png_set_unknown_chunk_location := GetProcAddr('png_set_unknown_chunk_location');
      png_get_unknown_chunks := GetProcAddr('png_get_unknown_chunks');
      png_handle_as_unknown := GetProcAddr('png_handle_as_unknown');
      png_set_invalid := GetProcAddr('png_set_invalid');
      png_read_png := GetProcAddr('png_read_png');
      png_write_png := GetProcAddr('png_write_png');
      png_sig_bytes := GetProcAddr('png_sig_bytes');
      png_get_copyright := GetProcAddr('png_get_copyright');
      png_get_header_ver := GetProcAddr('png_get_header_ver');
      png_get_header_version := GetProcAddr('png_get_header_version');
      png_get_libpng_ver := GetProcAddr('png_get_libpng_ver');
      png_permit_mng_features := GetProcAddr('png_permit_mng_features');
      png_get_mmx_flagmask := GetProcAddr('png_get_mmx_flagmask');
      png_get_asm_flagmask := GetProcAddr('png_get_asm_flagmask');
      png_get_asm_flags := GetProcAddr('png_get_asm_flags');
      png_get_mmx_bitdepth_threshold := GetProcAddr('png_get_mmx_bitdepth_threshold');
      png_get_mmx_rowbytes_threshold := GetProcAddr('png_get_mmx_rowbytes_threshold');
      png_set_asm_flags := GetProcAddr('png_set_asm_flags');
      png_set_mmx_thresholds := GetProcAddr('png_set_mmx_thresholds');
      png_mmx_support := GetProcAddr('png_mmx_support');
      png_set_strip_error_numbers := GetProcAddr('png_set_strip_error_numbers');
      png_set_user_limits := GetProcAddr('png_set_user_limits');
      png_get_user_width_max := GetProcAddr('png_get_user_width_max');
      png_get_user_height_max := GetProcAddr('png_get_user_height_max');
      png_get_uint_32 := GetProcAddr('png_get_uint_32');
      png_get_uint_16 := GetProcAddr('png_get_uint_16');
      png_get_int_32 := GetProcAddr('png_get_int_32');
      png_get_uint_31 := GetProcAddr('png_get_uint_31');
      png_save_uint_32 := GetProcAddr('png_save_uint_32');
      png_save_int_32 := GetProcAddr('png_save_int_32');
      png_save_uint_16 := GetProcAddr('png_save_uint_16');
      png_read_init := GetProcAddr('png_read_init');
      png_read_init_3 := GetProcAddr('png_read_init_3');
      png_read_init_2 := GetProcAddr('png_read_init_2');
      png_write_init := GetProcAddr('png_write_init');
      png_write_init_3 := GetProcAddr('png_write_init_3');
      png_write_init_2 := GetProcAddr('png_write_init_2');
    end;
  end;

  // check pointers
  Temp :=
    (libPNG_Handle <> (*$ifdef win32*) 0 (*$else*) nil (*$endif*)) ;
(*
    (Addr(png_access_version_number) <> nil) and
    (Addr(png_set_sig_bytes) <> nil) and
    (Addr(png_sig_cmp) <> nil) and
    (Addr(png_check_sig) <> nil) and
    (Addr(png_create_read_struct) <> nil) and
    (Addr(png_create_write_struct) <> nil) and
    (Addr(png_get_compression_buffer_size) <> nil) and
    (Addr(png_set_compression_buffer_size) <> nil) and
    (Addr(png_reset_zstream) <> nil) and
    (Addr(png_create_read_struct_2) <> nil) and
    (Addr(png_create_write_struct_2) <> nil) and
    (Addr(png_write_chunk) <> nil) and
    (Addr(png_write_chunk_start) <> nil) and
    (Addr(png_write_chunk_data) <> nil) and
    (Addr(png_write_chunk_end) <> nil) and
    (Addr(png_create_info_struct) <> nil) and
    (Addr(png_info_init) <> nil) and
    (Addr(png_info_init_3) <> nil) and
    (Addr(png_write_info_before_PLTE) <> nil) and
    (Addr(png_write_info) <> nil) and
    (Addr(png_read_info) <> nil) and
    (Addr(png_convert_to_rfc1123) <> nil) and
    (Addr(png_convert_from_struct_tm) <> nil) and
    (Addr(png_convert_from_time_t) <> nil) and
    (Addr(png_set_expand) <> nil) and
    (Addr(png_set_expand_gray_1_2_4_to_8) <> nil) and
    (Addr(png_set_palette_to_rgb) <> nil) and
    (Addr(png_set_tRNS_to_alpha) <> nil) and
    (Addr(png_set_gray_1_2_4_to_8) <> nil) and
    (Addr(png_set_bgr) <> nil) and
    (Addr(png_set_gray_to_rgb) <> nil) and
    (Addr(png_set_rgb_to_gray) <> nil) and
    (Addr(png_set_rgb_to_gray_fixed) <> nil) and
    (Addr(png_get_rgb_to_gray_status) <> nil) and
    (Addr(png_build_grayscale_palette) <> nil) and
    (Addr(png_set_strip_alpha) <> nil) and
    (Addr(png_set_swap_alpha) <> nil) and
    (Addr(png_set_invert_alpha) <> nil) and
    (Addr(png_set_filler) <> nil) and
    (Addr(png_set_add_alpha) <> nil) and
    (Addr(png_set_swap) <> nil) and
    (Addr(png_set_packing) <> nil) and
    (Addr(png_set_packswap) <> nil) and
    (Addr(png_set_shift) <> nil) and
    (Addr(png_set_interlace_handling) <> nil) and
    (Addr(png_set_invert_mono) <> nil) and
    (Addr(png_set_background) <> nil) and
    (Addr(png_set_strip_16) <> nil) and
    (Addr(png_set_dither) <> nil) and
    (Addr(png_set_gamma) <> nil) and
    (Addr(png_permit_empty_plte) <> nil) and
    (Addr(png_set_flush) <> nil) and
    (Addr(png_write_flush) <> nil) and
    (Addr(png_start_read_image) <> nil) and
    (Addr(png_read_update_info) <> nil) and
    (Addr(png_read_rows) <> nil) and
    (Addr(png_read_row) <> nil) and
    (Addr(png_read_image) <> nil) and
    (Addr(png_write_row) <> nil) and
    (Addr(png_write_rows) <> nil) and
    (Addr(png_write_image) <> nil) and
    (Addr(png_write_end) <> nil) and
    (Addr(png_read_end) <> nil) and
    (Addr(png_destroy_info_struct) <> nil) and
    (Addr(png_destroy_read_struct) <> nil) and
    (Addr(png_destroy_write_struct) <> nil) and
    (Addr(png_set_crc_action) <> nil) and
    (Addr(png_set_filter) <> nil) and
    (Addr(png_set_filter_heuristics) <> nil) and
    (Addr(png_set_compression_level) <> nil) and
    (Addr(png_set_compression_mem_level) <> nil) and
    (Addr(png_set_compression_strategy) <> nil) and
    (Addr(png_set_compression_window_bits) <> nil) and
    (Addr(png_set_compression_method) <> nil) and
    (Addr(png_init_io) <> nil) and
    (Addr(png_set_error_fn) <> nil) and
    (Addr(png_get_error_ptr) <> nil) and
    (Addr(png_set_write_fn) <> nil) and
    (Addr(png_set_read_fn) <> nil) and
    (Addr(png_get_io_ptr) <> nil) and
    (Addr(png_set_read_status_fn) <> nil) and
    (Addr(png_set_write_status_fn) <> nil) and
    (Addr(png_set_mem_fn) <> nil) and
    (Addr(png_get_mem_ptr) <> nil) and
    (Addr(png_set_read_user_transform_fn) <> nil) and
    (Addr(png_set_write_user_transform_fn) <> nil) and
    (Addr(png_set_user_transform_info) <> nil) and
    (Addr(png_get_user_transform_ptr) <> nil) and
    (Addr(png_set_read_user_chunk_fn) <> nil) and
    (Addr(png_get_user_chunk_ptr) <> nil) and
    (Addr(png_set_progressive_read_fn) <> nil) and
    (Addr(png_get_progressive_ptr) <> nil) and
    (Addr(png_process_data) <> nil) and
    (Addr(png_progressive_combine_row) <> nil) and
    (Addr(png_malloc) <> nil) and
    (Addr(png_malloc_warn) <> nil) and
    (Addr(png_free) <> nil) and
    (Addr(png_zalloc) <> nil) and
    (Addr(png_zfree) <> nil) and
    (Addr(png_free_data) <> nil) and
    (Addr(png_data_freer) <> nil) and
    (Addr(png_malloc_default) <> nil) and
    (Addr(png_free_default) <> nil) and
    (Addr(png_memcpy_check) <> nil) and
    (Addr(png_memset_check) <> nil) and
    (Addr(png_far_to_near) <> nil) and
    (Addr(png_error) <> nil) and
    (Addr(png_chunk_error) <> nil) and
    (Addr(png_warning) <> nil) and
    (Addr(png_chunk_warning) <> nil) and
    (Addr(png_get_valid) <> nil) and
    (Addr(png_get_rowbytes) <> nil) and
    (Addr(png_get_rows) <> nil) and
    (Addr(png_set_rows) <> nil) and
    (Addr(png_get_channels) <> nil) and
    (Addr(png_get_image_width) <> nil) and
    (Addr(png_get_image_height) <> nil) and
    (Addr(png_get_bit_depth) <> nil) and
    (Addr(png_get_color_type) <> nil) and
    (Addr(png_get_filter_type) <> nil) and
    (Addr(png_get_interlace_type) <> nil) and
    (Addr(png_get_compression_type) <> nil) and
    (Addr(png_get_pixels_per_meter) <> nil) and
    (Addr(png_get_x_pixels_per_meter) <> nil) and
    (Addr(png_get_y_pixels_per_meter) <> nil) and
    (Addr(png_get_pixel_aspect_ratio) <> nil) and
    (Addr(png_get_x_offset_pixels) <> nil) and
    (Addr(png_get_y_offset_pixels) <> nil) and
    (Addr(png_get_x_offset_microns) <> nil) and
    (Addr(png_get_y_offset_microns) <> nil) and
    (Addr(png_get_signature) <> nil) and
    (Addr(png_get_bKGD) <> nil) and
    (Addr(png_set_bKGD) <> nil) and
    (Addr(png_get_cHRM) <> nil) and
    (Addr(png_get_cHRM_fixed) <> nil) and
    (Addr(png_set_cHRM) <> nil) and
    (Addr(png_set_cHRM_fixed) <> nil) and
    (Addr(png_get_gAMA) <> nil) and
    (Addr(png_get_gAMA_fixed) <> nil) and
    (Addr(png_set_gAMA) <> nil) and
    (Addr(png_set_gAMA_fixed) <> nil) and
    (Addr(png_get_hIST) <> nil) and
    (Addr(png_set_hIST) <> nil) and
    (Addr(png_get_IHDR) <> nil) and
    (Addr(png_set_IHDR) <> nil) and
    (Addr(png_get_oFFs) <> nil) and
    (Addr(png_set_oFFs) <> nil) and
    (Addr(png_get_pCAL) <> nil) and
    (Addr(png_set_pCAL) <> nil) and
    (Addr(png_get_pHYs) <> nil) and
    (Addr(png_set_pHYs) <> nil) and
    (Addr(png_get_PLTE) <> nil) and
    (Addr(png_set_PLTE) <> nil) and
    (Addr(png_get_sBIT) <> nil) and
    (Addr(png_set_sBIT) <> nil) and
    (Addr(png_get_sRGB) <> nil) and
    (Addr(png_set_sRGB) <> nil) and
    (Addr(png_set_sRGB_gAMA_and_cHRM) <> nil) and
    (Addr(png_get_iCCP) <> nil) and
    (Addr(png_set_iCCP) <> nil) and
    (Addr(png_get_sPLT) <> nil) and
    (Addr(png_set_sPLT) <> nil) and
    (Addr(png_get_text) <> nil) and
    (Addr(png_set_text) <> nil) and
    (Addr(png_get_tIME) <> nil) and
    (Addr(png_set_tIME) <> nil) and
    (Addr(png_get_tRNS) <> nil) and
    (Addr(png_set_tRNS) <> nil) and
    (Addr(png_get_sCAL) <> nil) and
    (Addr(png_get_sCAL_s) <> nil) and
    (Addr(png_set_sCAL) <> nil) and
    (Addr(png_set_sCAL_s) <> nil) and
    (Addr(png_set_keep_unknown_chunks) <> nil) and
    (Addr(png_set_unknown_chunks) <> nil) and
    (Addr(png_set_unknown_chunk_location) <> nil) and
    (Addr(png_get_unknown_chunks) <> nil) and
    (Addr(png_handle_as_unknown) <> nil) and
    (Addr(png_set_invalid) <> nil) and
    (Addr(png_read_png) <> nil) and
    (Addr(png_write_png) <> nil) and
    (Addr(png_sig_bytes) <> nil) and
    (Addr(png_get_copyright) <> nil) and
    (Addr(png_get_header_ver) <> nil) and
    (Addr(png_get_header_version) <> nil) and
    (Addr(png_get_libpng_ver) <> nil) and
    (Addr(png_permit_mng_features) <> nil) and
    (Addr(png_get_mmx_flagmask) <> nil) and
    (Addr(png_get_asm_flagmask) <> nil) and
    (Addr(png_get_asm_flags) <> nil) and
    (Addr(png_get_mmx_bitdepth_threshold) <> nil) and
    (Addr(png_get_mmx_rowbytes_threshold) <> nil) and
    (Addr(png_set_asm_flags) <> nil) and
    (Addr(png_set_mmx_thresholds) <> nil) and
    (Addr(png_mmx_support) <> nil) and
    (Addr(png_set_strip_error_numbers) <> nil) and
    (Addr(png_set_user_limits) <> nil) and
    (Addr(png_get_user_width_max) <> nil) and
    (Addr(png_get_user_height_max) <> nil) and
    (Addr(png_get_uint_32) <> nil) and
    (Addr(png_get_uint_16) <> nil) and
    (Addr(png_get_int_32) <> nil) and
    (Addr(png_get_uint_31) <> nil) and
    (Addr(png_save_uint_32) <> nil) and
    (Addr(png_save_int_32) <> nil) and
    (Addr(png_save_uint_16) <> nil) and
    (Addr(png_read_init) <> nil) and
    (Addr(png_read_init_3) <> nil) and
    (Addr(png_read_init_2) <> nil) and
    (Addr(png_write_init) <> nil) and
    (Addr(png_write_init_3) <> nil) and
    (Addr(png_write_init_2) <> nil);
*)

  if Temp then
    Inc(libPNG_RefCount);

  Result := Temp;
end;


procedure quit_libPNG;
begin
  Dec(libPNG_RefCount);

  if libPNG_RefCount <= 0 then begin
    if libPNG_Handle <> (*$ifdef win32*) 0 (*$else*) nil (*$endif*) then begin
      (*$ifdef win32*)
        FreeLibrary(libPNG_Handle);
        libPNG_Handle := 0;
      (*$else*)
        dlclose(libPNG_Handle);
        libPNG_Handle := nil;
      (*$endif*)
    end;

    png_access_version_number := nil;
    png_set_sig_bytes := nil;
    png_sig_cmp := nil;
    png_check_sig := nil;
    png_create_read_struct := nil;
    png_create_write_struct := nil;
    png_get_compression_buffer_size := nil;
    png_set_compression_buffer_size := nil;
    png_reset_zstream := nil;
    png_create_read_struct_2 := nil;
    png_create_write_struct_2 := nil;
    png_write_chunk := nil;
    png_write_chunk_start := nil;
    png_write_chunk_data := nil;
    png_write_chunk_end := nil;
    png_create_info_struct := nil;
    png_info_init := nil;
    png_info_init_3 := nil;
    png_write_info_before_PLTE := nil;
    png_write_info := nil;
    png_read_info := nil;
    png_convert_to_rfc1123 := nil;
    png_convert_from_struct_tm := nil;
    png_convert_from_time_t := nil;
    png_set_expand := nil;
    png_set_expand_gray_1_2_4_to_8 := nil;
    png_set_palette_to_rgb := nil;
    png_set_tRNS_to_alpha := nil;
    png_set_gray_1_2_4_to_8 := nil;
    png_set_bgr := nil;
    png_set_gray_to_rgb := nil;
    png_set_rgb_to_gray := nil;
    png_set_rgb_to_gray_fixed := nil;
    png_get_rgb_to_gray_status := nil;
    png_build_grayscale_palette := nil;
    png_set_strip_alpha := nil;
    png_set_swap_alpha := nil;
    png_set_invert_alpha := nil;
    png_set_filler := nil;
    png_set_add_alpha := nil;
    png_set_swap := nil;
    png_set_packing := nil;
    png_set_packswap := nil;
    png_set_shift := nil;
    png_set_interlace_handling := nil;
    png_set_invert_mono := nil;
    png_set_background := nil;
    png_set_strip_16 := nil;
    png_set_dither := nil;
    png_set_gamma := nil;
    png_permit_empty_plte := nil;
    png_set_flush := nil;
    png_write_flush := nil;
    png_start_read_image := nil;
    png_read_update_info := nil;
    png_read_rows := nil;
    png_read_row := nil;
    png_read_image := nil;
    png_write_row := nil;
    png_write_rows := nil;
    png_write_image := nil;
    png_write_end := nil;
    png_read_end := nil;
    png_destroy_info_struct := nil;
    png_destroy_read_struct := nil;
    png_destroy_write_struct := nil;
    png_set_crc_action := nil;
    png_set_filter := nil;
    png_set_filter_heuristics := nil;
    png_set_compression_level := nil;
    png_set_compression_mem_level := nil;
    png_set_compression_strategy := nil;
    png_set_compression_window_bits := nil;
    png_set_compression_method := nil;
    png_init_io := nil;
    png_set_error_fn := nil;
    png_get_error_ptr := nil;
    png_set_write_fn := nil;
    png_set_read_fn := nil;
    png_get_io_ptr := nil;
    png_set_read_status_fn := nil;
    png_set_write_status_fn := nil;
    png_set_mem_fn := nil;
    png_get_mem_ptr := nil;
    png_set_read_user_transform_fn := nil;
    png_set_write_user_transform_fn := nil;
    png_set_user_transform_info := nil;
    png_get_user_transform_ptr := nil;
    png_set_read_user_chunk_fn := nil;
    png_get_user_chunk_ptr := nil;
    png_set_progressive_read_fn := nil;
    png_get_progressive_ptr := nil;
    png_process_data := nil;
    png_progressive_combine_row := nil;
    png_malloc := nil;
    png_malloc_warn := nil;
    png_free := nil;
    png_zalloc := nil;
    png_zfree := nil;
    png_free_data := nil;
    png_data_freer := nil;
    png_malloc_default := nil;
    png_free_default := nil;
    png_memcpy_check := nil;
    png_memset_check := nil;
    png_far_to_near := nil;
    png_error := nil;
    png_chunk_error := nil;
    png_warning := nil;
    png_chunk_warning := nil;
    png_get_valid := nil;
    png_get_rowbytes := nil;
    png_get_rows := nil;
    png_set_rows := nil;
    png_get_channels := nil;
    png_get_image_width := nil;
    png_get_image_height := nil;
    png_get_bit_depth := nil;
    png_get_color_type := nil;
    png_get_filter_type := nil;
    png_get_interlace_type := nil;
    png_get_compression_type := nil;
    png_get_pixels_per_meter := nil;
    png_get_x_pixels_per_meter := nil;
    png_get_y_pixels_per_meter := nil;
    png_get_pixel_aspect_ratio := nil;
    png_get_x_offset_pixels := nil;
    png_get_y_offset_pixels := nil;
    png_get_x_offset_microns := nil;
    png_get_y_offset_microns := nil;
    png_get_signature := nil;
    png_get_bKGD := nil;
    png_set_bKGD := nil;
    png_get_cHRM := nil;
    png_get_cHRM_fixed := nil;
    png_set_cHRM := nil;
    png_set_cHRM_fixed := nil;
    png_get_gAMA := nil;
    png_get_gAMA_fixed := nil;
    png_set_gAMA := nil;
    png_set_gAMA_fixed := nil;
    png_get_hIST := nil;
    png_set_hIST := nil;
    png_get_IHDR := nil;
    png_set_IHDR := nil;
    png_get_oFFs := nil;
    png_set_oFFs := nil;
    png_get_pCAL := nil;
    png_set_pCAL := nil;
    png_get_pHYs := nil;
    png_set_pHYs := nil;
    png_get_PLTE := nil;
    png_set_PLTE := nil;
    png_get_sBIT := nil;
    png_set_sBIT := nil;
    png_get_sRGB := nil;
    png_set_sRGB := nil;
    png_set_sRGB_gAMA_and_cHRM := nil;
    png_get_iCCP := nil;
    png_set_iCCP := nil;
    png_get_sPLT := nil;
    png_set_sPLT := nil;
    png_get_text := nil;
    png_set_text := nil;
    png_get_tIME := nil;
    png_set_tIME := nil;
    png_get_tRNS := nil;
    png_set_tRNS := nil;
    png_get_sCAL := nil;
    png_get_sCAL_s := nil;
    png_set_sCAL := nil;
    png_set_sCAL_s := nil;
    png_set_keep_unknown_chunks := nil;
    png_set_unknown_chunks := nil;
    png_set_unknown_chunk_location := nil;
    png_get_unknown_chunks := nil;
    png_handle_as_unknown := nil;
    png_set_invalid := nil;
    png_read_png := nil;
    png_write_png := nil;
    png_sig_bytes := nil;
    png_get_copyright := nil;
    png_get_header_ver := nil;
    png_get_header_version := nil;
    png_get_libpng_ver := nil;
    png_permit_mng_features := nil;
    png_get_mmx_flagmask := nil;
    png_get_asm_flagmask := nil;
    png_get_asm_flags := nil;
    png_get_mmx_bitdepth_threshold := nil;
    png_get_mmx_rowbytes_threshold := nil;
    png_set_asm_flags := nil;
    png_set_mmx_thresholds := nil;
    png_mmx_support := nil;
    png_set_strip_error_numbers := nil;
    png_set_user_limits := nil;
    png_get_user_width_max := nil;
    png_get_user_height_max := nil;
    png_get_uint_32 := nil;
    png_get_uint_16 := nil;
    png_get_int_32 := nil;
    png_get_uint_31 := nil;
    png_save_uint_32 := nil;
    png_save_int_32 := nil;
    png_save_uint_16 := nil;
    png_read_init := nil;
    png_read_init_3 := nil;
    png_read_init_2 := nil;
    png_write_init := nil;
    png_write_init_3 := nil;
    png_write_init_2 := nil;
  end;
end;


end.